use crate::{message_exchange::rabbitmq::EXCHANGE_NAME_JOB_RESPONSE, Result};
use amq_protocol::types::LongLongUInt;
use lapin::{
  message::Delivery,
  options::{BasicAckOptions, BasicPublishOptions, BasicRejectOptions},
  BasicProperties, Channel,
};
use std::sync::Arc;

pub async fn publish_job_response(
  channel: Arc<Channel>,
  deliveries: Vec<Delivery>,
  queue_name: &str,
  payload: &str,
) -> Result<()> {
  log::debug!("Job publish {}", payload);
  let result = channel
    .basic_publish(
      EXCHANGE_NAME_JOB_RESPONSE,
      queue_name,
      BasicPublishOptions::default(),
      payload.as_bytes().to_vec(),
      BasicProperties::default(),
    )
    .wait()
    .is_ok();

  if result {
    for delivery in deliveries {
      ack(channel.clone(), delivery.delivery_tag).await?;
    }
  } else {
    for delivery in deliveries {
      reject(channel.clone(), delivery.delivery_tag).await?;
    }
  }

  Ok(())
}

async fn ack(channel: Arc<Channel>, delivery_tag: LongLongUInt) -> Result<()> {
  log::debug!("Ack delivery ack {}", delivery_tag);
  channel
    .basic_ack(
      delivery_tag,
      BasicAckOptions::default(), /*not requeue*/
    )
    .await
    .map_err(|e| e.into())
}

async fn reject(channel: Arc<Channel>, delivery_tag: LongLongUInt) -> Result<()> {
  log::debug!("Reject delivery ack {}", delivery_tag);
  channel
    .basic_reject(
      delivery_tag,
      BasicRejectOptions { requeue: true }, /*requeue*/
    )
    .await
    .map_err(|e| e.into())
}
