use amq_protocol_uri::{AMQPAuthority, AMQPScheme, AMQPUri, AMQPUserInfo};
use std::env;

macro_rules! get_env_value {
  ($key:expr, $default:expr) => {
    match env::var($key) {
      Ok(value) => value,
      _ => $default.to_string(),
    }
  };
}
const AMQP_HOSTNAME: &str = "AMQP_HOSTNAME";
const AMQP_PORT: &str = "AMQP_PORT";
const AMQP_USERNAME: &str = "AMQP_USERNAME";
const AMQP_PASSWORD: &str = "AMQP_PASSWORD";
const AMQP_VHOST: &str = "AMQP_VHOST";
const AMQP_VIRTUAL_HOST: &str = "AMQP_VIRTUAL_HOST";
const AMQP_QUEUE: &str = "AMQP_QUEUE";
const AMQP_TLS: &str = "AMQP_TLS";
const AMQP_TLS_CERT: &str = "AMQP_TLS_CERT";

const MESSAGE_TTL_LIMIT: &str = "MESSAGE_TTL_LIMIT";
const MESSAGE_SOFT_LIMIT: &str = "MESSAGE_SOFT_LIMIT";

const FFMPEG_TTL_LIMIT: &str = "FFMPEG_TTL_LIMIT";

const SOURCE_ORDERS: &str = "SOURCE_ORDERS";

fn get_amqp_tls() -> bool {
  let value = get_env_value!(AMQP_TLS, "true");
  matches!(value.as_str(), "true" | "1" | "True" | "TRUE")
}

fn get_amqp_tls_certificate_path() -> Option<String> {
  env::var(AMQP_TLS_CERT).ok()
}

pub fn get_amqp_tls_certificate() -> Option<String> {
  get_amqp_tls_certificate_path()
    .and_then(|certificate_path| std::fs::read_to_string(certificate_path).ok())
}

fn get_amqp_hostname() -> String {
  get_env_value!(AMQP_HOSTNAME, "127.0.0.1")
}

fn get_amqp_port() -> u16 {
  let value = get_env_value!(AMQP_PORT, "5672");
  match value.parse::<u16>() {
    Ok(value) => value,
    _ => 5672,
  }
}

fn get_amqp_username() -> String {
  get_env_value!(AMQP_USERNAME, "guest")
}

fn get_amqp_password() -> String {
  get_env_value!(AMQP_PASSWORD, "guest")
}

fn get_amqp_vhost() -> String {
  get_env_value!(AMQP_VHOST, get_env_value!(AMQP_VIRTUAL_HOST, "/"))
}

pub fn get_amqp_queue() -> String {
  get_env_value!(AMQP_QUEUE, "job_undefined")
}

pub fn get_store_hostname(store_code: &str) -> String {
  get_store_hostname_with_default(store_code, "http://127.0.0.1:4000/api")
}

pub fn get_store_hostname_with_default(store_code: &str, default: &str) -> String {
  get_env_value!(&format!("{}_HOSTNAME", store_code), default)
}

pub fn get_store_username(store_code: &str) -> String {
  get_env_value!(&format!("{}_USERNAME", store_code), "")
}

pub fn get_store_password(store_code: &str) -> String {
  get_env_value!(&format!("{}_PASSWORD", store_code), "")
}

pub fn get_store_token(store_code: &str) -> String {
  get_env_value!(&format!("{}_TOKEN", store_code), "")
}

pub fn get_x_death_count_limit() -> i64 {
  let value = get_env_value!(MESSAGE_TTL_LIMIT, "4");
  match value.parse::<i64>() {
    Ok(value) => value,
    _ => 4,
  }
}

pub fn get_soft_reject_count_limit() -> i64 {
  let value = get_env_value!(MESSAGE_SOFT_LIMIT, "800");
  match value.parse::<i64>() {
    Ok(value) => value,
    _ => 800,
  }
}

pub fn get_ffmpeg_waitmore_limit() -> i64 {
  let value = get_env_value!(FFMPEG_TTL_LIMIT, "1_000_000");
  match value.parse::<i64>() {
    Ok(value) => value,
    _ => 1_000_000,
  }
}

pub fn get_amqp_uri() -> AMQPUri {
  let amqp_tls = get_amqp_tls();
  let amqp_hostname = get_amqp_hostname();
  let amqp_port = get_amqp_port();
  let amqp_username = get_amqp_username();
  let amqp_password = get_amqp_password();
  let amqp_vhost = get_amqp_vhost();
  let amqp_queue = get_amqp_queue();

  log::info!("Start connection with configuration:");
  log::info!("AMQP TLS: {}", amqp_tls);
  if amqp_tls {
    log::info!("AMQP TLS CERT: {:?}", get_amqp_tls_certificate_path());
  }
  log::info!("AMQP HOSTNAME: {}", amqp_hostname);
  log::info!("AMQP PORT: {}", amqp_port);
  log::info!("AMQP USERNAME: {}", amqp_username);
  log::info!("AMQP VIRTUAL HOST: {}", amqp_vhost);
  log::info!("AMQP QUEUE: {}", amqp_queue);

  let scheme = if amqp_tls {
    AMQPScheme::AMQPS
  } else {
    AMQPScheme::AMQP
  };

  AMQPUri {
    scheme,
    authority: AMQPAuthority {
      userinfo: AMQPUserInfo {
        username: amqp_username,
        password: amqp_password,
      },
      host: amqp_hostname,
      port: amqp_port,
    },
    vhost: amqp_vhost,
    query: Default::default(),
  }
}

pub fn get_source_orders() -> Option<Vec<String>> {
  env::var(SOURCE_ORDERS)
    .map(|source_orders| {
      Some(
        source_orders
          .split(':')
          .map(|path| path.to_string())
          .collect(),
      )
    })
    .unwrap_or(None)
}

#[test]
fn configuration() {
  fn test_config<T: std::cmp::PartialEq + std::fmt::Debug>(
    func: fn() -> T,
    expected_value: T,
    env_var: &str,
  ) {
    // Store and remove environment variable
    let env_value = env::var_os(env_var);
    env::remove_var(env_var);

    // Compare values
    assert_eq!(func(), expected_value);

    // Reset environment variable
    if let Some(env_value) = env_value {
      env::set_var(env_var, env_value)
    }
  }

  test_config(get_amqp_hostname, "127.0.0.1".to_string(), AMQP_HOSTNAME);
  test_config(get_amqp_port, 5672, AMQP_PORT);
  test_config(get_amqp_username, "guest".to_string(), AMQP_USERNAME);
  test_config(get_amqp_password, "guest".to_string(), AMQP_PASSWORD);
  test_config(get_amqp_queue, "job_undefined".to_string(), AMQP_QUEUE);

  assert!(get_store_hostname("BACKEND") == *"http://127.0.0.1:4000/api");
  assert!(get_store_username("BACKEND") == *"");
  assert!(get_store_password("BACKEND") == *"");

  env::set_var("AMQP_TLS", "1");
  assert!(get_amqp_tls());
  env::set_var("AMQP_TLS", "False");
  assert!(!get_amqp_tls());
  env::set_var("AMQP_VHOST", "/");
  assert!(get_amqp_vhost() == *"/");
  env::set_var("AMQP_PORT", "BAD_VALUE");
  assert!(get_amqp_port() == 5672);

  env::set_var("MESSAGE_TTL_LIMIT", "2");
  assert!(get_x_death_count_limit() == 2);
  env::set_var("MESSAGE_SOFT_LIMIT", "1000");
  assert!(get_soft_reject_count_limit() == 1000);

  env::set_var("FFMPEG_TTL_LIMIT", "10000000");
  assert!(get_ffmpeg_waitmore_limit() == 10000000);
}
