use assert_matches::assert_matches;
use mcai_worker_sdk::prelude::*;
use std::collections::HashMap;

#[test]
pub fn test_retrieving_vault_credentials() {
  const VAULT_TOKEN: &str = "VAULT_TOKEN";

  if let Err(_error) = std::env::var(VAULT_TOKEN) {
    eprintln!("{} environment variable not set. Skip test.", VAULT_TOKEN);
    return;
  }

  struct Worker {}

  #[derive(Clone, Debug, Deserialize)]
  struct VaultCredentials {
    data: HashMap<String, String>,
  }

  #[derive(Clone, Debug, Deserialize, JsonSchema)]
  pub struct WorkerParameters {
    credentials: HashMap<String, String>,
  }

  impl McaiWorker<WorkerParameters> for Worker {
    fn get_name(&self) -> String {
      "Vault credentials test Worker ".to_string()
    }

    fn get_short_description(&self) -> String {
      "The Worker defined in Vault credentials functional tests".to_string()
    }

    fn get_description(&self) -> String {
      "Mock a Worker to check the SDK allow workers to retrieve properly credentials from Vault"
        .to_string()
    }

    fn get_version(&self) -> semver::Version {
      semver::Version::parse("1.2.3").unwrap()
    }

    fn init(&mut self) -> Result<()> {
      log::info!("Initialize Vault credentials test worker!");
      Ok(())
    }

    fn process(
      &self,
      _channel: Option<McaiChannel>,
      parameters: WorkerParameters,
      job_result: JobResult,
    ) -> Result<JobResult>
    where
      Self: std::marker::Sized,
    {
      // Read expected credentials from file
      let file = std::fs::File::open("./tests/functional/vault_credentials.json").unwrap();
      let expected_credentials: VaultCredentials = serde_json::from_reader(file).unwrap();

      for (expected_key, expected_value) in expected_credentials.data.iter() {
        let value = parameters.credentials.get(expected_key).ok_or_else(|| {
          MessageError::RuntimeError(format!(
            "Could not find expected credential: '{}'",
            expected_key
          ))
        })?;

        log::debug!("Found credentials: {}={}", expected_key, value);

        if value != expected_value {
          return Err(MessageError::RuntimeError(format!(
            "Invalid credential value: '{}' instead of '{}'",
            value, expected_value
          )));
        }
      }

      Ok(job_result)
    }
  }

  let message = r#"{
    "job_id": 666,
    "parameters": [
      {
        "id": "credentials",
        "type": "string",
        "store": "VAULT",
        "value": "test"
      }
    ]
  }"#;

  let worker = Worker {};
  let worker_configuration = WorkerConfiguration::new("", &worker, "instance_id").unwrap();
  let worker = Arc::new(Mutex::new(worker));

  let local_exchange = LocalExchange::new();
  let mut local_exchange = Arc::new(local_exchange);
  let exchange = local_exchange.clone();

  std::thread::spawn(move || {
    let processor = Processor::new(exchange, worker_configuration);
    assert!(processor.run(worker).is_ok());
  });

  let local_exchange = Arc::make_mut(&mut local_exchange);

  let response = local_exchange.next_response().unwrap();
  assert_matches!(response.unwrap(), ResponseMessage::WorkerCreated(_));

  let job = Job::new(message).unwrap();
  local_exchange.send_order(OrderMessage::Job(job)).unwrap();

  let response = local_exchange.next_response().unwrap();
  assert_matches!(response.unwrap(), ResponseMessage::WorkerStarted(_));

  let response = local_exchange.next_response().unwrap();
  assert_matches!(
    response.unwrap(),
    ResponseMessage::Feedback(Feedback::Progression(JobProgression {
      job_id: 666,
      progression: 0,
      ..
    }))
  );

  let response = local_exchange.next_response().unwrap();
  assert_matches!(
    response.unwrap(),
    ResponseMessage::Completed(JobResult { .. })
  );
}
