use super::{publish, CurrentOrders};
use crate::{
  message_exchange::{
    message::{Feedback, ResponseMessage},
    rabbitmq::current_orders::OrderDeliveryType,
  },
  MessageError, Result,
};
use async_std::{
  channel::{self, Receiver, Sender},
  sync::{Arc, Mutex as AsyncMutex},
  task::{self, JoinHandle},
};
use lapin::Channel;
use std::sync::Mutex;

pub struct RabbitmqPublisher {
  handle: Option<JoinHandle<()>>,
  response_sender: Sender<ResponseMessage>,
}

impl RabbitmqPublisher {
  pub async fn new(channel: &Channel, current_orders: Arc<Mutex<CurrentOrders>>) -> Result<Self> {
    let (response_sender, response_receiver) = channel::unbounded();

    let response_receiver = Arc::new(AsyncMutex::new(response_receiver));

    let channel = Arc::new(channel.clone());

    let handle = Some(task::spawn(async move {
      loop {
        if let Err(error) = Self::handle_response(
          response_receiver.clone(),
          channel.clone(),
          current_orders.clone(),
        )
        .await
        {
          log::error!("{:?}", error);
        }
      }
    }));

    Ok(RabbitmqPublisher {
      handle,
      response_sender,
    })
  }

  pub async fn send_response(&self, response: ResponseMessage) {
    self.response_sender.send(response).await.unwrap();
  }

  async fn respond_with_delivery(
    channel: Arc<Channel>,
    current_orders: Arc<Mutex<CurrentOrders>>,
    order_types: Vec<OrderDeliveryType>,
    response: ResponseMessage,
  ) {
    let mut unacknowledged_deliveries = vec![];
    for order_type in order_types {
      if let Some(unacknowledged_delivery) = current_orders
        .lock()
        .unwrap()
        .get_order_delivery_for_type(order_type.clone())
        .unwrap()
        .get_unacknowledged_delivery()
      {
        unacknowledged_deliveries.push(unacknowledged_delivery);
      }
    }

    match publish::response_with_delivery(
      channel.clone(),
      unacknowledged_deliveries.clone(),
      &response,
    )
    .await
    {
      Ok(_) => {
        for unacknowledged_delivery in unacknowledged_deliveries {
          current_orders
            .lock()
            .unwrap()
            .get_order_delivery_for_tag(unacknowledged_delivery.delivery_tag)
            .unwrap()
            .set_acknowledged()
        }
      }
      Err(error) => {
        if let Err(error) = publish::error(channel.clone(), unacknowledged_deliveries, &error).await
        {
          log::error!("Unable to publish response: {:?}", error);
        }
      }
    }
  }

  async fn handle_response(
    response_receiver: Arc<AsyncMutex<Receiver<ResponseMessage>>>,
    channel: Arc<Channel>,
    current_orders: Arc<Mutex<CurrentOrders>>,
  ) -> Result<()> {
    let response = response_receiver.lock().await.recv().await.map_err(|e| {
      MessageError::RuntimeError(format!(
        "unable to wait response from processor: {:?}",
        e.to_string()
      ))
    })?;

    log::debug!("Response: {:?}", response);
    log::debug!("{}", current_orders.lock().unwrap());

    match &response {
      ResponseMessage::Feedback(Feedback::Progression(progression)) => {
        publish::job_progression(channel, progression.clone())?;
      }
      ResponseMessage::WorkerCreated(_) => {
        publish::response_with_delivery(channel.clone(), vec![], &response.clone()).await?;
      }
      ResponseMessage::WorkerTerminated(_) => {
        if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Kill)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Kill],
            response.clone(),
          )
          .await;
        } else {
          publish::response_with_delivery(channel.clone(), vec![], &response.clone()).await?;
        }
      }
      ResponseMessage::WorkerInitialized(_) => {
        if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Init)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Init],
            response.clone(),
          )
          .await;
        } else if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Job)
        {
          Self::respond_with_delivery(channel, current_orders.clone(), vec![], response.clone())
            .await;
        }
      }
      ResponseMessage::WorkerStarted(_) => {
        if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Start)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Start],
            response.clone(),
          )
          .await;
        } else if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Job)
        {
          Self::respond_with_delivery(channel, current_orders.clone(), vec![], response.clone())
            .await;
        }
      }
      ResponseMessage::WorkerUpdated(_) => {
        if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Update)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Update],
            response.clone(),
          )
          .await;
        } else if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Job)
        {
          Self::respond_with_delivery(channel, current_orders.clone(), vec![], response.clone())
            .await;
        }
      }
      ResponseMessage::JobStopped(_) => {
        let mut order_types = vec![];

        if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Stop)
        {
          order_types.push(OrderDeliveryType::Stop);
        }

        if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Job)
        {
          order_types.push(OrderDeliveryType::Job);
        }

        if !order_types.is_empty() {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            order_types,
            response.clone(),
          )
          .await
        }
      }
      ResponseMessage::Completed(_) | ResponseMessage::Error(_) => {
        if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Job)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Job],
            response.clone(),
          )
          .await;
        } else if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Stop)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Stop],
            response.clone(),
          )
          .await;
        } else if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::ReachedExpiration)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::ReachedExpiration],
            response.clone(),
          )
          .await;
        }
      }
      ResponseMessage::Feedback(_) | ResponseMessage::StatusError(_) => {
        if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Suspend)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Suspend],
            response.clone(),
          )
          .await;
        } else if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Resume)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Resume],
            response.clone(),
          )
          .await;
        } else if current_orders
          .lock()
          .unwrap()
          .has_order_delivery_for_type(OrderDeliveryType::Status)
        {
          Self::respond_with_delivery(
            channel,
            current_orders.clone(),
            vec![OrderDeliveryType::Status],
            response.clone(),
          )
          .await;
        }
      }
    }

    match response {
      ResponseMessage::WorkerCreated(_)
      | ResponseMessage::WorkerInitialized(_)
      | ResponseMessage::WorkerStarted(_)
      | ResponseMessage::WorkerUpdated(_)
      | ResponseMessage::WorkerTerminated(_) => {}
      ResponseMessage::Completed(_)
      | ResponseMessage::Error(_)
      | ResponseMessage::JobStopped(_) => {
        current_orders.lock().unwrap().reset_process_deliveries();
      }
      ResponseMessage::Feedback(_) | ResponseMessage::StatusError(_) => {
        current_orders.lock().unwrap().reset_status_deliveries();
      }
    };

    Ok(())
  }
}

impl Drop for RabbitmqPublisher {
  fn drop(&mut self) {
    self.handle.take().map(JoinHandle::cancel);
  }
}
