mod threaded_media_process;

use crate::{
  job::{JobResult, JobStatus},
  message_exchange::message::{Feedback, OrderMessage, ResponseMessage},
  processor::{Process, ProcessStatus},
  worker::{WorkerActivity, WorkerConfiguration, WorkerStatus},
  McaiChannel, MessageError, MessageEvent, Result,
};
use schemars::JsonSchema;
use serde::de::DeserializeOwned;
use std::{
  cell::RefCell,
  rc::Rc,
  sync::{
    mpsc::{channel, Sender},
    Arc, Mutex,
  },
};
use threaded_media_process::ThreadedMediaProcess;

pub struct MediaProcess {
  order_sender: Sender<OrderMessage>,
  current_job_result: Arc<Mutex<Option<JobResult>>>,
}

impl<P: DeserializeOwned + JsonSchema, ME: 'static + MessageEvent<P> + Send> Process<P, ME>
  for MediaProcess
{
  fn new(
    message_event: Arc<Mutex<ME>>,
    response_sender: McaiChannel,
    worker_configuration: WorkerConfiguration,
  ) -> Self {
    let (order_sender, order_receiver) = channel();

    let worker_activity = Arc::new(Mutex::new(WorkerActivity::Idle));
    let job_status = Arc::new(Mutex::new(JobStatus::Unknown));
    let current_job_result = Arc::new(Mutex::new(None));
    let cloned_current_job_result = current_job_result.clone();

    std::thread::spawn(move || {
      let mut threaded_media_process: Option<Rc<RefCell<ThreadedMediaProcess>>> = None;

      let mut keep_running = true;

      // let mut received = order_receiver.recv();

      while let Ok(message) = &order_receiver.recv() {
        // Process the received order message
        let response = match message {
          OrderMessage::Job(job) => {
            if let Err(error) = job.check_requirements() {
              ResponseMessage::Error(error)
            } else {
              log::info!("Process job: {:?}", job);
              let initialization_result = ThreadedMediaProcess::initialize_process(
                message_event.clone(),
                job.clone(),
                worker_activity.clone(),
              );

              if let Err(error) = initialization_result {
                Self::update_status(
                  job_status.clone(),
                  worker_activity.clone(),
                  Some(JobStatus::Error),
                  None,
                );
                ResponseMessage::Error(error)
              } else {
                threaded_media_process =
                  Some(Rc::new(RefCell::new(initialization_result.unwrap())));

                *current_job_result.lock().unwrap() = Some(
                  threaded_media_process
                    .clone()
                    .unwrap()
                    .borrow()
                    .job_result
                    .clone(),
                );

                // TODO send worker response Initialized
                Self::update_status(
                  job_status.clone(),
                  worker_activity.clone(),
                  Some(JobStatus::Running),
                  None,
                );

                let response = threaded_media_process
                  .clone()
                  .unwrap()
                  .borrow_mut()
                  .start_process(
                    message_event.clone(),
                    &order_receiver,
                    response_sender.clone(),
                    worker_configuration.clone(),
                  );

                log::debug!("Finished response: {:?}", response);

                let status = Self::get_job_status_from_response(
                  response.clone(),
                  job_status.lock().unwrap().clone(),
                );

                Self::update_status(
                  job_status.clone(),
                  worker_activity.clone(),
                  Some(status),
                  None,
                );

                *current_job_result.lock().unwrap() = None;

                response
              }
            }
          }
          OrderMessage::InitProcess(job) => {
            if let Err(error) = job.check_requirements() {
              ResponseMessage::Error(error)
            } else {
              let initialization_result = ThreadedMediaProcess::initialize_process(
                message_event.clone(),
                job.clone(),
                worker_activity.clone(),
              );

              if let Err(error) = initialization_result {
                Self::update_status(
                  job_status.clone(),
                  worker_activity.clone(),
                  Some(JobStatus::Error),
                  None,
                );
                ResponseMessage::Error(error)
              } else {
                Self::update_status(
                  job_status.clone(),
                  worker_activity.clone(),
                  Some(JobStatus::Initialized),
                  None,
                );

                threaded_media_process =
                  Some(Rc::new(RefCell::new(initialization_result.unwrap())));

                *current_job_result.lock().unwrap() = Some(
                  threaded_media_process
                    .clone()
                    .unwrap()
                    .borrow()
                    .job_result
                    .clone(),
                );

                ResponseMessage::WorkerInitialized(
                  JobResult::new(job.job_id).with_status(JobStatus::Initialized),
                )
              }
            }
          }
          OrderMessage::StartProcess(job) => {
            Self::update_status(
              job_status.clone(),
              worker_activity.clone(),
              Some(JobStatus::Running),
              None,
            );

            let response = if let Some(media_process_parameters) = &threaded_media_process {
              media_process_parameters.borrow_mut().start_process(
                message_event.clone(),
                &order_receiver,
                response_sender.clone(),
                worker_configuration.clone(),
              )
            } else {
              ResponseMessage::Error(MessageError::ProcessingError(
                JobResult::new(job.job_id)
                  .with_status(JobStatus::Error)
                  .with_message("Process cannot be started, it must be initialized before!"),
              ))
            };

            log::debug!("Finished response: {:?}", response);

            let status = Self::get_job_status_from_response(
              response.clone(),
              job_status.lock().unwrap().clone(),
            );

            Self::update_status(
              job_status.clone(),
              worker_activity.clone(),
              Some(status),
              None,
            );

            response
          }
          OrderMessage::UpdateProcess(job) => {
            ResponseMessage::Error(MessageError::ProcessingError(
              JobResult::new(job.job_id)
                .with_status(JobStatus::Error)
                .with_message("Cannot update a non-running job."),
            ))
          }
          OrderMessage::StopProcess(job) => ResponseMessage::Error(MessageError::ProcessingError(
            JobResult::new(job.job_id)
              .with_status(JobStatus::Error)
              .with_message("Cannot stop a non-running job."),
          )),
          OrderMessage::Status => Self::get_status_feedback(
            current_job_result.lock().unwrap().clone(),
            job_status.lock().unwrap().clone(),
            worker_activity.lock().unwrap().clone(),
            worker_configuration.clone(),
          ),
          OrderMessage::StopWorker => {
            keep_running = false;
            Self::get_status_feedback(
              current_job_result.lock().unwrap().clone(),
              job_status.lock().unwrap().clone(),
              worker_activity.lock().unwrap().clone(),
              worker_configuration.clone(),
            )
          }
          OrderMessage::StopConsumingJobs => {
            Self::update_status(
              job_status.clone(),
              worker_activity.clone(),
              None,
              Some(WorkerActivity::Suspended),
            );
            Self::get_status_feedback(
              current_job_result.lock().unwrap().clone(),
              job_status.lock().unwrap().clone(),
              worker_activity.lock().unwrap().clone(),
              worker_configuration.clone(),
            )
          }
          OrderMessage::ResumeConsumingJobs => {
            Self::update_status(
              job_status.clone(),
              worker_activity.clone(),
              None,
              Some(WorkerActivity::Idle),
            );
            Self::get_status_feedback(
              current_job_result.lock().unwrap().clone(),
              job_status.lock().unwrap().clone(),
              worker_activity.lock().unwrap().clone(),
              worker_configuration.clone(),
            )
          }
          OrderMessage::ReachedExpiration(job) => {
            Self::update_status(
              Arc::new(Mutex::new(JobStatus::Error)),
              worker_activity.clone(),
              None,
              None,
            );
            ResponseMessage::Error(MessageError::ProcessingError(
              JobResult::new(job.job_id)
                .with_status(JobStatus::Error)
                .with_message("Job TTL has been reached."),
            ))
          }
        };

        match response {
          ResponseMessage::Completed(_)
          | ResponseMessage::Error(_)
          | ResponseMessage::JobStopped(_) => {
            *current_job_result.lock().unwrap() = None;
          }
          _ => {}
        }

        // Send the action response
        log::trace!("Send the action response message");
        response_sender
          .lock()
          .unwrap()
          .send_response(response)
          .unwrap();

        // If the process is stopped, stop looping
        if !keep_running {
          break;
        }
      }
    });

    MediaProcess {
      order_sender,
      current_job_result: cloned_current_job_result,
    }
  }

  fn handle(&mut self, _message_event: Arc<Mutex<ME>>, order_message: OrderMessage) -> Result<()> {
    if let Err(error) = self.order_sender.send(order_message) {
      return Err(MessageError::RuntimeError(error.to_string())); // TODO use ProcessError
    }
    Ok(())
  }

  fn get_current_job_id(&self, _message_event: Arc<Mutex<ME>>) -> Option<u64> {
    self
      .current_job_result
      .lock()
      .unwrap()
      .clone()
      .map(|job_result| job_result.get_job_id())
  }
}

impl MediaProcess {
  fn update_status(
    current_job_status: Arc<Mutex<JobStatus>>,
    current_worker_activity: Arc<Mutex<WorkerActivity>>,
    job_status: Option<JobStatus>,
    worker_activity: Option<WorkerActivity>,
  ) {
    match (job_status, worker_activity) {
      (Some(status), Some(activity)) => {
        *current_job_status.lock().unwrap() = status;
        *current_worker_activity.lock().unwrap() = activity;
      }
      (Some(status), None) => {
        *current_job_status.lock().unwrap() = status;

        // Update worker activity if not suspended
        if *current_worker_activity.lock().unwrap() != WorkerActivity::Suspended {
          *current_worker_activity.lock().unwrap() =
            current_job_status.lock().unwrap().clone().into();
        }
      }
      (None, Some(activity)) => {
        *current_worker_activity.lock().unwrap() = activity;
      }
      (None, None) => {}
    }
  }

  fn get_job_status_from_response(response: ResponseMessage, default: JobStatus) -> JobStatus {
    match response {
      ResponseMessage::Completed(_) => JobStatus::Completed,
      ResponseMessage::JobStopped(_) => JobStatus::Stopped,
      ResponseMessage::Error(_) => JobStatus::Error,
      ResponseMessage::Feedback(_)
      | ResponseMessage::StatusError(_)
      | ResponseMessage::WorkerCreated(_)
      | ResponseMessage::WorkerInitialized(_)
      | ResponseMessage::WorkerStarted(_)
      | ResponseMessage::WorkerUpdated(_)
      | ResponseMessage::WorkerTerminated(_) => {
        log::error!(
          "Should not have to handle such a message as a processing response: {:?}",
          response
        );
        default
      }
    }
  }

  fn get_status_feedback(
    job_result: Option<JobResult>,
    job_status: JobStatus,
    worker_activity: WorkerActivity,
    worker_configuration: WorkerConfiguration,
  ) -> ResponseMessage {
    let job_result = job_result.map(|job_result| job_result.with_status(job_status.clone()));

    // Check whether the worker activity is coherent with the job status
    let worker_activity = match worker_activity {
      WorkerActivity::Idle | WorkerActivity::Busy => {
        if worker_activity != WorkerActivity::from(job_status.clone()) {
          WorkerActivity::from(job_status)
        } else {
          worker_activity
        }
      }
      WorkerActivity::Suspended => worker_activity,
    };

    ResponseMessage::Feedback(Feedback::Status(ProcessStatus::new(
      WorkerStatus::new(worker_activity, worker_configuration),
      job_result,
    )))
  }
}
