use crate::client::AmqpConnection;
use lapin::types::{AMQPValue, FieldTable};
use mcai_worker_sdk::prelude::*;
use std::collections::BTreeMap;
use std::sync::mpsc;

#[async_std::test]
async fn rabbitmq_ttl_job() -> Result<()> {
  struct Worker {}

  #[derive(Clone, Debug, Deserialize, JsonSchema)]
  pub struct WorkerParameters {}

  impl MessageEvent<WorkerParameters> for Worker {
    fn get_name(&self) -> String {
      "Test Worker".to_string()
    }

    fn get_short_description(&self) -> String {
      "The Worker defined in unit tests".to_string()
    }

    fn get_description(&self) -> String {
      "Mock a Worker to realise tests around SDK".to_string()
    }

    fn get_version(&self) -> semver::Version {
      semver::Version::parse("1.2.3").unwrap()
    }

    fn init(&mut self) -> Result<()> {
      log::info!("Initialize processor test worker!");
      Ok(())
    }

    fn process(
      &self,
      channel: Option<McaiChannel>,
      _parameters: WorkerParameters,
      job_result: JobResult,
    ) -> Result<JobResult>
    where
      Self: std::marker::Sized,
    {
      assert!(channel.is_some());
      return Ok(job_result.with_status(JobStatus::Completed));
    }
  }

  let (created_sender, created_receiver) = mpsc::channel::<WorkerConfiguration>();
  let (status_sender, status_receiver) = mpsc::channel::<ProcessStatus>();
  let (error_sender, error_receiver) = mpsc::channel::<JobResult>();

  let amqp_connection = AmqpConnection::new().unwrap();

  amqp_connection.start_consumer(QUEUE_WORKER_CREATED, created_sender, 1);
  amqp_connection.start_consumer(QUEUE_WORKER_STATUS, status_sender, 1);
  amqp_connection.start_consumer(QUEUE_JOB_ERROR, error_sender, 3);

  let instance_id = "908126354";
  let worker = Worker {};
  let worker_configuration = WorkerConfiguration::new("job_test", &worker, instance_id).unwrap();
  let rabbitmq_exchange = RabbitmqExchange::new(&worker_configuration).await;

  if let Err(MessageError::Amqp(lapin::Error::IOError(error))) = rabbitmq_exchange {
    eprintln!(
      "Connection to RabbitMQ failure: {}. Skip test.",
      error.to_string()
    );
    return Ok(());
  }

  let rabbitmq_exchange = Arc::new(rabbitmq_exchange.unwrap());

  let cloned_worker_configuration = worker_configuration.clone();

  let worker = Arc::new(Mutex::new(worker));

  std::thread::spawn(move || {
    let processor = Processor::new(rabbitmq_exchange, cloned_worker_configuration);
    assert!(processor.run(worker).is_ok());
  });

  assert!(created_receiver.recv().is_ok());

  amqp_connection.send_order(vec![instance_id], &OrderMessage::Status)?;
  assert!(status_receiver.recv().is_ok());

  let job = Job::new(
    r#"{
    "job_id": 123,
    "parameters": [
      { "id":"requirements",
        "type":"requirements",
        "value": {
          "paths": [
            "nonexistent_file"
          ]
        }
      }
    ]
  }"#,
  )
  .unwrap();

  amqp_connection.send_order_to_queue(
    "job_test",
    FieldTable::default(),
    &OrderMessage::ReachedExpiration(job.clone()),
  )?;

  assert!(error_receiver.recv().is_ok());

  let mut map = FieldTable::from(BTreeMap::new());
  let mut properties = FieldTable::from(BTreeMap::new());
  properties.insert("count".into(), AMQPValue::LongLongInt(5));

  map.insert(
    "x-death".into(),
    AMQPValue::FieldArray(vec![AMQPValue::FieldTable(properties)].into()),
  );

  amqp_connection.send_order_to_queue(
    "job_test",
    map.clone(),
    &OrderMessage::StartProcess(job.clone()),
  )?;

  assert!(error_receiver.recv().is_ok());

  amqp_connection.send_order_to_queue(
    "job_test",
    FieldTable::default(),
    &OrderMessage::Job(job.clone()),
  )?;

  assert!(error_receiver.recv().is_ok());

  Ok(())
}
