use crate::client::AmqpConnection;
use mcai_worker_sdk::prelude::*;
use serial_test::serial;
use std::sync::{
  mpsc::{self, Sender},
  Arc, Mutex,
};

#[async_std::test]
#[serial]
async fn update_job() -> Result<()> {
  let file_path = "./test_rabbitmq_media_processor_update_job.mxf";
  let nb_frames = 500;
  super::ffmpeg::create_xdcam_sample_file(file_path, nb_frames).unwrap();

  struct Worker {
    updatable: String,
  }

  #[derive(Clone, Debug, Deserialize, JsonSchema)]
  pub struct WorkerParameters {
    source_path: String,
    destination_path: String,
    updatable: String,
  }

  impl MessageEvent<WorkerParameters> for Worker {
    fn get_name(&self) -> String {
      "Test Worker".to_string()
    }

    fn get_short_description(&self) -> String {
      "The Worker defined in unit tests".to_string()
    }

    fn get_description(&self) -> String {
      "Mock a Worker to realise tests around SDK".to_string()
    }

    fn get_version(&self) -> semver::Version {
      semver::Version::parse("1.2.3").unwrap()
    }

    fn init(&mut self) -> Result<()> {
      log::info!("Update processor test worker!");
      Ok(())
    }

    fn init_process(
      &mut self,
      parameters: WorkerParameters,
      format_context: Arc<Mutex<FormatContext>>,
      _result: Arc<Mutex<Sender<ProcessResult>>>,
    ) -> Result<Vec<StreamDescriptor>> {
      self.updatable = parameters.updatable;
      let mut stream_descriptors = vec![];

      let format_context = format_context.lock().unwrap();
      for stream_index in 0..format_context.get_nb_streams() {
        let stream_type = format_context.get_stream_type(stream_index as isize);
        info!(
          "Handle stream #{} with type: {:?}",
          stream_index, stream_type
        );

        match stream_type {
          AVMediaType::AVMEDIA_TYPE_VIDEO => {
            let filters = vec![VideoFilter::Resize(Scaling {
              width: Some(200),
              height: Some(70),
            })];
            stream_descriptors.push(StreamDescriptor::new_video(stream_index as usize, filters))
          }
          AVMediaType::AVMEDIA_TYPE_AUDIO => {
            let channel_layouts = vec!["mono".to_string()];
            let sample_formats = vec!["s16".to_string()];
            let sample_rates = vec![16000];

            let filters = vec![AudioFilter::Format(AudioFormat {
              sample_rates,
              channel_layouts,
              sample_formats,
            })];
            stream_descriptors.push(StreamDescriptor::new_audio(stream_index as usize, filters))
          }
          AVMediaType::AVMEDIA_TYPE_SUBTITLE => {
            stream_descriptors.push(StreamDescriptor::new_data(stream_index as usize))
          }
          AVMediaType::AVMEDIA_TYPE_DATA => {
            stream_descriptors.push(StreamDescriptor::new_data(stream_index as usize))
          }
          _ => info!("Skip stream #{}", stream_index),
        };
      }
      Ok(stream_descriptors)
    }

    fn process_frame(
      &mut self,
      _job_result: JobResult,
      _stream_index: usize,
      _frame: ProcessFrame,
    ) -> Result<ProcessResult> {
      log::info!("Process frame");
      Ok(ProcessResult::new_json(""))
    }

    fn update_process(&mut self, parameters: WorkerParameters) -> Result<()> {
      self.updatable = parameters.updatable;
      Ok(())
    }
  }

  let (created_sender, created_receiver) = mpsc::channel::<WorkerConfiguration>();
  let (status_sender, status_receiver) = mpsc::channel::<ProcessStatus>();
  let (initialized_sender, initialized_receiver) = mpsc::channel::<JobResult>();
  let (started_sender, started_receiver) = mpsc::channel::<JobResult>();
  let (updated_sender, updated_receiver) = mpsc::channel::<JobResult>();

  let (progression_sender, progression_receiver) = mpsc::channel::<JobProgression>();
  let (stopped_sender, stopped_receiver) = mpsc::channel::<JobResult>();

  let amqp_connection = AmqpConnection::new().unwrap();

  amqp_connection.start_consumer(QUEUE_WORKER_CREATED, created_sender, 1);
  amqp_connection.start_consumer(QUEUE_WORKER_STATUS, status_sender, 1);
  amqp_connection.start_consumer(QUEUE_WORKER_INITIALIZED, initialized_sender, 1);
  amqp_connection.start_consumer(QUEUE_WORKER_STARTED, started_sender, 1);
  amqp_connection.start_consumer(QUEUE_WORKER_UPDATED, updated_sender, 1);

  amqp_connection.start_consumer(QUEUE_JOB_PROGRESSION, progression_sender, 1);
  amqp_connection.start_consumer(QUEUE_JOB_STOPPED, stopped_sender, 1);

  let instance_id = "update_job_9876543210";
  let worker = Worker {
    updatable: "".to_string(),
  };
  let worker_configuration = WorkerConfiguration::new("", &worker, instance_id).unwrap();
  let rabbitmq_exchange = RabbitmqExchange::new(&worker_configuration).await;

  if let Err(MessageError::Amqp(lapin::Error::IOError(error))) = rabbitmq_exchange {
    eprintln!(
      "Connection to RabbitMQ failure: {}. Skip test.",
      error.to_string()
    );
    return Ok(());
  }

  let rabbitmq_exchange = Arc::new(rabbitmq_exchange.unwrap());

  let worker = Arc::new(Mutex::new(worker));

  let cl_worker = Arc::clone(&worker);

  std::thread::spawn(move || {
    let processor = Processor::new(rabbitmq_exchange, worker_configuration);
    assert!(processor.run(cl_worker).is_ok());
  });

  assert!(created_receiver.recv().is_ok());

  amqp_connection.send_order(vec![instance_id], &OrderMessage::Status)?;
  assert!(status_receiver.recv().is_ok());

  let job = Job::new(
    r#"{
    "job_id": 999,
    "parameters": [
      {
        "id": "source_path",
        "type": "string",
        "value": "./test_rabbitmq_media_processor_update_job.mxf"
      },
      {
        "id": "destination_path",
        "type": "string",
        "value": "./test_rabbitmq_media_processor_update_job.json"
      },
      {
        "id": "updatable",
        "type": "string",
        "value": "basic"
      }
    ]
  }"#,
  )
  .unwrap();

  amqp_connection.send_order(vec![instance_id], &OrderMessage::InitProcess(job.clone()))?;
  assert!(initialized_receiver.recv().is_ok());

  amqp_connection.send_order(vec![instance_id], &OrderMessage::StartProcess(job.clone()))?;

  assert!(started_receiver.recv().is_ok());
  assert!(progression_receiver.recv().is_ok());

  let updated_job = Job::new(
    r#"{
    "job_id": 999,
    "parameters": [
      {
        "id": "source_path",
        "type": "string",
        "value": "./test_rabbitmq_media_processor_update_job.mxf"
      },
      {
        "id": "destination_path",
        "type": "string",
        "value": "./test_rabbitmq_media_processor_update_job.json"
      },
      {
        "id": "updatable",
        "type": "string",
        "value": "updated"
      }
    ]
  }"#,
  )
  .unwrap();

  amqp_connection.send_order(vec![instance_id], &OrderMessage::UpdateProcess(updated_job))?;
  let updated_message = updated_receiver.recv();
  assert!(updated_message.is_ok());

  let updated_param: String = worker.lock().unwrap().updatable.clone();
  assert_eq!(updated_param, "updated".to_string());

  amqp_connection.send_order(vec![instance_id], &OrderMessage::StopProcess(job))?;
  let stopped_message = stopped_receiver.recv();
  assert!(stopped_message.is_ok());

  std::thread::sleep(std::time::Duration::from_millis(2000));

  log::info!("RabbitMQ update test done!");

  std::fs::remove_file(file_path).unwrap();

  Ok(())
}
