use crate::{
  job::{Job, JobProgression, JobResult, JobStatus},
  message::media::{
    finish_process, initialize_process,
    output::Output,
    source::{DecodeResult, Source},
  },
  message_exchange::message::{Feedback, OrderMessage, ResponseMessage},
  processor::ProcessStatus,
  publish_job_progression,
  worker::{WorkerActivity, WorkerConfiguration, WorkerStatus},
  McaiChannel, MessageError, MessageEvent, Result,
};
use schemars::JsonSchema;
use serde::de::DeserializeOwned;
use std::sync::{mpsc::Receiver, Arc, Mutex};

pub struct ThreadedMediaProcess {
  source: Source,
  output: Output,
  keep_running: bool,
  pub job: Job,
  worker_activity: Arc<Mutex<WorkerActivity>>,
  pub job_result: JobResult,
}

impl ThreadedMediaProcess {
  pub fn initialize_process<
    P: DeserializeOwned + JsonSchema,
    ME: 'static + MessageEvent<P> + Send,
  >(
    message_event: Arc<Mutex<ME>>,
    job: Job,
    worker_activity: Arc<Mutex<WorkerActivity>>,
  ) -> Result<Self> {
    log::info!("Initialize job: {:?}", job);

    let job_result = JobResult::new(job.job_id);
    initialize_process(message_event, &job).map(|(source, output)| ThreadedMediaProcess {
      source,
      output,
      keep_running: false,
      job,
      worker_activity,
      job_result,
    })
  }

  pub fn get_status_feedback(
    &self,
    status: JobStatus,
    worker_configuration: WorkerConfiguration,
  ) -> ResponseMessage {
    ResponseMessage::Feedback(Feedback::Status(
      self.get_process_status(status, worker_configuration),
    ))
  }

  fn get_process_status(
    &self,
    status: JobStatus,
    worker_configuration: WorkerConfiguration,
  ) -> ProcessStatus {
    let job_result = self.job_result.clone().with_status(status.clone());

    let worker_activity = self.worker_activity.lock().unwrap().clone();

    // Check whether the worker activity is coherent with the job status
    let worker_activity = match worker_activity {
      WorkerActivity::Idle | WorkerActivity::Busy => {
        if worker_activity != WorkerActivity::from(status.clone()) {
          WorkerActivity::from(status)
        } else {
          worker_activity
        }
      }
      WorkerActivity::Suspended => worker_activity,
    };

    let worker_status = WorkerStatus::new(worker_activity, worker_configuration);
    ProcessStatus::new(worker_status, Some(job_result))
  }

  pub fn start_process<P: DeserializeOwned + JsonSchema, ME: 'static + MessageEvent<P> + Send>(
    &mut self,
    message_event: Arc<Mutex<ME>>,
    order_receiver: &Receiver<OrderMessage>,
    response_sender: McaiChannel,
    worker_configuration: WorkerConfiguration,
  ) -> ResponseMessage {
    let job = self.job.clone();
    self.job_result = self.job_result.clone().with_status(JobStatus::Running);

    response_sender
      .lock()
      .unwrap()
      .send_response(ResponseMessage::WorkerStarted(self.job_result.clone()))
      .unwrap();

    log::info!("Start processing job: {:?}", job);

    // start publishing progression
    response_sender
      .lock()
      .unwrap()
      .send_response(ResponseMessage::Feedback(Feedback::Progression(
        JobProgression::new(job.job_id, 0),
      )))
      .unwrap();

    log::info!(
      "{} - Start to process media (start: {} ms, duration: {})",
      self.job_result.get_str_job_id(),
      self.source.get_start_offset(),
      self
        .source
        .get_segment_duration()
        .map(|duration| format!("{} ms", duration))
        .unwrap_or_else(|| "unknown".to_string())
    );

    let process_duration_ms = self.source.get_segment_duration();

    let mut processed_frames = 0;
    let mut previous_progress = 0;

    let first_stream_fps = self
      .source
      .get_stream_fps(self.source.get_first_stream_index()) as f32;

    loop {
      if response_sender.lock().unwrap().is_stopped() {
        log::info!("Stopped !");
        break ResponseMessage::JobStopped(self.job_result.clone().with_status(JobStatus::Stopped));
      }

      // Process next frame
      let response = self
        .process_frame(
          message_event.clone(),
          response_sender.clone(),
          first_stream_fps,
          process_duration_ms,
          &mut processed_frames,
          &mut previous_progress,
        )
        .unwrap_or_else(|error| Some(ResponseMessage::Error(error)));

      // If a message is returned, stop looping and forward the message
      if let Some(message) = response {
        break message;
      }

      // Otherwise check whether an order message as been sent to this thread
      if let Ok(message) = order_receiver.try_recv() {
        let resp = match message {
          OrderMessage::Job(_) => ResponseMessage::Error(MessageError::ProcessingError(
            self
              .job_result
              .clone()
              .with_status(JobStatus::Running)
              .with_message("Cannot handle a job while a process is running"),
          )),
          OrderMessage::InitProcess(_) => ResponseMessage::Error(MessageError::ProcessingError(
            self
              .job_result
              .clone()
              .with_status(JobStatus::Running)
              .with_message("Cannot initialize a running process"),
          )),
          OrderMessage::StartProcess(_) => ResponseMessage::Error(MessageError::ProcessingError(
            self
              .job_result
              .clone()
              .with_status(JobStatus::Running)
              .with_message("Cannot start a running process"),
          )),
          OrderMessage::UpdateProcess(job) => self.update_process(message_event.clone(), job),
          OrderMessage::StopProcess(_) => {
            // Stop the process properly
            break finish_process(message_event, &mut self.output, self.job_result.clone())
              .map(ResponseMessage::JobStopped)
              .unwrap_or_else(ResponseMessage::Error);
          }
          OrderMessage::Status => {
            self.get_status_feedback(JobStatus::Running, worker_configuration.clone())
          }
          OrderMessage::StopWorker => {
            // Kill the process
            self.keep_running = false;
            self.get_status_feedback(JobStatus::Stopped, worker_configuration.clone())
          }
          OrderMessage::StopConsumingJobs => {
            // If this message comes here, it means a process is running
            *self.worker_activity.lock().unwrap() = WorkerActivity::Suspended;
            self.get_status_feedback(JobStatus::Running, worker_configuration.clone())
          }
          OrderMessage::ResumeConsumingJobs => {
            // If this message comes here, it means a process is running
            *self.worker_activity.lock().unwrap() = WorkerActivity::Busy;
            self.get_status_feedback(JobStatus::Running, worker_configuration.clone())
          }
          OrderMessage::ReachedExpiration(job) => {
            ResponseMessage::Error(MessageError::ProcessingError(
              JobResult::new(job.job_id)
                .with_status(JobStatus::Error)
                .with_message("Job TTL has been reached."),
            ))
          }
        };

        response_sender.lock().unwrap().send_response(resp).unwrap();
      }
    }
  }

  pub fn update_process<P: DeserializeOwned + JsonSchema, ME: 'static + MessageEvent<P> + Send>(
    &mut self,
    message_event: Arc<Mutex<ME>>,
    job: Job,
  ) -> ResponseMessage {
    let parameters = job.get_parameters().unwrap();

    log::info!("Start updating job: {:?}", job);

    // Update process
    let result = message_event.lock().unwrap().update_process(parameters);

    if let Err(e) = result {
      return ResponseMessage::WorkerUpdated(
        JobResult::new(job.job_id)
          .with_status(JobStatus::Error)
          .with_message(&format!("Update error: {:?}", e)),
      );
    }

    let job_result = self.job_result.clone();
    ResponseMessage::WorkerUpdated(job_result.with_status(JobStatus::Updated))
  }

  #[allow(clippy::too_many_arguments)]
  pub fn process_frame<P: DeserializeOwned + JsonSchema, ME: 'static + MessageEvent<P> + Send>(
    &mut self,
    message_event: Arc<Mutex<ME>>,
    response_sender: McaiChannel,
    first_stream_fps: f32,
    process_duration_ms: Option<u64>,
    processed_frames: &mut usize,
    previous_progress: &mut u8,
  ) -> Result<Option<ResponseMessage>> {
    let response = match self.source.next_frame()? {
      DecodeResult::Frame {
        stream_index,
        frame,
      } => {
        if stream_index == self.source.get_first_stream_index() {
          (*processed_frames) += 1;

          let processed_ms = (*processed_frames) as f32 * 1000.0 / first_stream_fps;

          if let Some(duration) = process_duration_ms {
            let progress = std::cmp::min((processed_ms / duration as f32 * 100.0) as u8, 100);
            if progress > (*previous_progress) {
              publish_job_progression(
                Some(response_sender),
                self.job_result.get_job_id(),
                progress,
              )?;
              (*previous_progress) = progress;
            }
          }
        }
        log::info!(
          "{} - Process frame {}",
          self.job_result.get_str_job_id(),
          processed_frames
        );

        let _process_result = crate::message::media::process_frame(
          message_event,
          &mut self.output,
          self.job_result.clone(),
          stream_index,
          frame,
        )?;

        None
      }
      DecodeResult::WaitMore | DecodeResult::Nothing => None,
      DecodeResult::EndOfStream => {
        log::debug!("Media Process: End Of Stream");
        let response = finish_process(message_event, &mut self.output, self.job_result.clone())
          .map(ResponseMessage::Completed)?;

        Some(response)
      }
    };

    Ok(response)
  }
}
