use lapin::message::Delivery;
use std::fmt::{Display, Formatter};

#[derive(Clone, Debug, PartialEq)]
pub(crate) enum OrderDeliveryType {
  Init,
  Job,
  Kill,
  Start,
  Stop,
  Status,
}

#[derive(Clone, Debug, PartialEq)]
pub(crate) struct OrderDelivery {
  pub(crate) delivery: Delivery,
  pub(crate) order_type: OrderDeliveryType,
  pub(crate) acknowledged: bool,
}

impl OrderDelivery {
  pub(crate) fn new(delivery: Delivery, order_type: OrderDeliveryType) -> Self {
    Self {
      delivery,
      order_type,
      acknowledged: false,
    }
  }

  pub(crate) fn get_unacknowledged_delivery(&self) -> Option<Delivery> {
    if self.acknowledged {
      return None;
    }
    Some(self.delivery.clone())
  }

  pub(crate) fn set_acknowledged(&mut self) {
    if self.acknowledged {
      log::warn!(
        "Try to acknowledge an already acknowledged delivery message: {}",
        self
      );
      return;
    }
    self.acknowledged = true;
  }

  fn has_order_type(&self, order_type: OrderDeliveryType) -> bool {
    self.order_type == order_type
  }
}

impl Display for OrderDelivery {
  fn fmt(&self, f: &mut Formatter<'_>) -> std::result::Result<(), std::fmt::Error> {
    write!(
      f,
      "order: {:?}, delivery: {}, acknowledged: {}",
      self.order_type, self.delivery.delivery_tag, self.acknowledged
    )
  }
}

#[derive(Clone, Debug, Default, PartialEq)]
pub struct CurrentOrders {
  pub(crate) orders: Vec<OrderDelivery>,
}

impl CurrentOrders {
  pub(crate) fn reset_process_deliveries(&mut self) {
    self.orders = self
      .orders
      .iter()
      .cloned()
      .filter(|order_delivery| {
        order_delivery.has_order_type(OrderDeliveryType::Status)
          || order_delivery.has_order_type(OrderDeliveryType::Kill)
      })
      .collect()
  }

  pub(crate) fn has_order_delivery_for(&self, order_type: OrderDeliveryType) -> bool {
    self
      .orders
      .iter()
      .any(|order_delivery| order_delivery.has_order_type(order_type.clone()))
  }

  pub(crate) fn get_order_delivery_for(
    &mut self,
    order_type: OrderDeliveryType,
  ) -> Option<&mut OrderDelivery> {
    self
      .orders
      .iter_mut()
      .find(|order_delivery| order_delivery.has_order_type(order_type.clone()))
  }

  pub(crate) fn reset_status_deliveries(&mut self) {
    self.orders = self
      .orders
      .iter()
      .cloned()
      .filter(|order_delivery| !order_delivery.has_order_type(OrderDeliveryType::Status))
      .collect()
  }
}

impl Display for CurrentOrders {
  fn fmt(&self, f: &mut Formatter<'_>) -> std::result::Result<(), std::fmt::Error> {
    let orders_as_string: Vec<String> = self
      .orders
      .iter()
      .map(|order_delivery| format!("{}", order_delivery))
      .collect();
    write!(f, "CurrentOrders: {:?}", orders_as_string)
  }
}

#[test]
pub fn test_current_orders() {
  let delivery = Delivery {
    delivery_tag: 0,
    exchange: Default::default(),
    routing_key: Default::default(),
    redelivered: false,
    properties: Default::default(),
    data: vec![],
    acker: Default::default(),
  };

  let order_delivery = OrderDelivery::new(delivery.clone(), OrderDeliveryType::Status);

  assert!(!order_delivery.acknowledged);
  assert_eq!(delivery, order_delivery.delivery);
  assert_eq!(OrderDeliveryType::Status, order_delivery.order_type);

  let mut current_orders = CurrentOrders::default();
  current_orders.orders.push(order_delivery);

  assert!(current_orders.has_order_delivery_for(OrderDeliveryType::Status));

  let extracted_order_delivery = current_orders.get_order_delivery_for(OrderDeliveryType::Status);

  assert!(extracted_order_delivery.is_some());

  let extracted_order_delivery = extracted_order_delivery.unwrap();
  assert!(!extracted_order_delivery.acknowledged);
  assert_eq!(delivery, extracted_order_delivery.delivery);
  assert_eq!(
    Some(delivery.clone()),
    extracted_order_delivery.get_unacknowledged_delivery()
  );
  assert_eq!(
    OrderDeliveryType::Status,
    extracted_order_delivery.order_type
  );

  extracted_order_delivery.set_acknowledged();
  assert!(extracted_order_delivery.acknowledged);
  assert_eq!(None, extracted_order_delivery.get_unacknowledged_delivery());

  current_orders
    .orders
    .push(OrderDelivery::new(delivery, OrderDeliveryType::Job));

  assert!(current_orders.has_order_delivery_for(OrderDeliveryType::Status));
  assert!(current_orders.has_order_delivery_for(OrderDeliveryType::Job));

  current_orders.reset_process_deliveries();

  assert!(current_orders.has_order_delivery_for(OrderDeliveryType::Status));
  assert!(!current_orders.has_order_delivery_for(OrderDeliveryType::Job));

  current_orders.reset_status_deliveries();

  assert!(!current_orders.has_order_delivery_for(OrderDeliveryType::Status));
  assert!(!current_orders.has_order_delivery_for(OrderDeliveryType::Job));

  assert!(current_orders.orders.is_empty());
}
