//! Workflow Definition
use crate::common::{Icon, Identifier, Notification, Parameter, Right, StartParameter, Step};
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
#[serde(tag = "schema_version")]
/// # Schema of Workflow Definition for Media Cloud AI
pub enum WorkflowDefinition {
  #[serde(rename = "1.8")]
  Version1_8(Version1_8),
  #[serde(rename = "1.9")]
  Version1_9(Version1_9),
  #[serde(rename = "1.10")]
  Version1_10(Version1_10),
}

impl WorkflowDefinition {
  pub fn new(identifier: &str) -> Self {
    WorkflowDefinition::Version1_10(Version1_10 {
      common: Common {
        icon: Icon { icon: None },
        identifier: Identifier {
          content: identifier.to_string(),
        },
        label: "".to_string(),
        is_live: false,
        tags: vec![],
        version_major: 0,
        version_minor: 0,
        version_micro: 1,
      },
      notifications: Notifications {
        notification_hooks: vec![],
      },
      parameters: Parameters { parameters: vec![] },
      steps: Steps { steps: vec![] },
      start_parameters: StartParameters {
        start_parameters: vec![],
      },
    })
  }

  pub fn identifier(&self) -> &str {
    match self {
      WorkflowDefinition::Version1_8(workflow) => &workflow.common.identifier.content,
      WorkflowDefinition::Version1_9(workflow) => &workflow.common.identifier.content,
      WorkflowDefinition::Version1_10(workflow) => &workflow.common.identifier.content,
    }
  }

  pub fn steps(&self) -> &Vec<Step> {
    match self {
      WorkflowDefinition::Version1_8(workflow) => &workflow.steps.steps,
      WorkflowDefinition::Version1_9(workflow) => &workflow.steps.steps,
      WorkflowDefinition::Version1_10(workflow) => &workflow.steps.steps,
    }
  }

  pub fn get_mut_steps(&mut self) -> &mut Vec<Step> {
    match self {
      WorkflowDefinition::Version1_8(workflow) => &mut workflow.steps.steps,
      WorkflowDefinition::Version1_9(workflow) => &mut workflow.steps.steps,
      WorkflowDefinition::Version1_10(workflow) => &mut workflow.steps.steps,
    }
  }
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
#[serde(deny_unknown_fields)]
pub struct Version1_8 {
  #[serde(flatten)]
  pub common: Common,
  #[serde(flatten)]
  pub parameters: Parameters,
  #[serde(flatten)]
  pub rights: Rights,
  #[serde(flatten)]
  pub steps: Steps,
  #[serde(flatten)]
  pub start_parameters: StartParameters,
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
#[serde(deny_unknown_fields)]
pub struct Version1_9 {
  #[serde(flatten)]
  pub common: Common,
  #[serde(flatten)]
  pub parameters: Parameters,
  #[serde(flatten)]
  pub steps: Steps,
  #[serde(flatten)]
  pub start_parameters: StartParameters,
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
#[serde(deny_unknown_fields)]
pub struct Version1_10 {
  #[serde(flatten)]
  pub common: Common,
  #[serde(flatten)]
  pub notifications: Notifications,
  #[serde(flatten)]
  pub parameters: Parameters,
  #[serde(flatten)]
  pub steps: Steps,
  #[serde(flatten)]
  pub start_parameters: StartParameters,
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
pub struct Common {
  /// The icon used with the label. One of [theses](https://material.io/resources/icons/) icons.
  pub icon: Icon,
  /// The Identifier of the workflow, used to reference it
  pub identifier: Identifier,
  /// The label of the workflow, used as displayed name
  pub label: String,
  /// Mentions if it defines a live workflow
  #[serde(default, skip_serializing_if = "crate::is_false")]
  pub is_live: bool,
  /// List of tags to classify the worklow
  #[serde(default, skip_serializing_if = "Vec::is_empty")]
  pub tags: Vec<String>,
  /// Major version of this Workflow definition
  pub version_major: u32,
  /// Minor version of this Workflow definition
  pub version_minor: u32,
  /// Micro version of this Workflow definition
  pub version_micro: u32,
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
pub struct Notifications {
  /// Define Notifications got this workflow (default: [])
  #[serde(default, skip_serializing_if = "Vec::is_empty")]
  pub notification_hooks: Vec<Notification>,
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
pub struct Rights {
  /// Defines rights for this definition
  #[serde(default, skip_serializing_if = "Vec::is_empty")]
  pub rights: Vec<Right>,
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
pub struct Parameters {
  /// Storage of dynamic parameters during process of the workflow instance
  pub parameters: Vec<Parameter>,
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
pub struct StartParameters {
  /// Definition of available parameters to start the workflow definition
  #[serde(default, skip_serializing_if = "Vec::is_empty")]
  pub start_parameters: Vec<StartParameter>,
}

#[derive(Clone, Debug, Deserialize, JsonSchema, PartialEq, Serialize)]
pub struct Steps {
  /// Defines rights for this definition
  pub steps: Vec<Step>,
}

#[cfg(test)]
mod tests {
  use super::WorkflowDefinition;

  #[test]
  fn load_version_1_8() {
    let str_workflow_definition =
      include_str!("../tests/resources/workflow_definitions/1.8/simple.json");
    let _: WorkflowDefinition = serde_json::from_str(str_workflow_definition).unwrap();

    let str_workflow_definition =
      include_str!("../tests/resources/workflow_definitions/1.8/transfer.json");
    let _: WorkflowDefinition = serde_json::from_str(str_workflow_definition).unwrap();
  }

  #[test]
  fn load_version_1_9() {
    let str_workflow_definition =
      include_str!("../tests/resources/workflow_definitions/1.9/objet_detection.json");
    let _: WorkflowDefinition = serde_json::from_str(str_workflow_definition).unwrap();
  }

  #[test]
  fn load_version_1_10() {
    let str_workflow_definition =
      include_str!("../tests/resources/workflow_definitions/1.10/simple.json");
    let _: WorkflowDefinition = serde_json::from_str(str_workflow_definition).unwrap();
  }
}
