use crate::subcommands::get_required_parameter_as_string;
use crate::{
  backend::Action,
  configuration::Configuration,
  constant::*,
  datetime::parse_date,
  error::Result,
  get_confirmation, models, print_response,
  subcommands::{
    get_command_option, get_multiple_command_options, get_required_parameters_as_integers,
  },
  GlobalFlags, Output,
};
use chrono::{DateTime, Utc};
use clap::{App, Arg, ArgMatches, SubCommand};
use std::convert::TryFrom;

#[derive(Debug)]
pub enum Workflow {
  Create {
    backend: String,
    output: Output,
  },
  List {
    after_date: Option<DateTime<Utc>>,
    backend: String,
    before_date: Option<DateTime<Utc>>,
    name: Option<String>,
    output: Output,
    states: Vec<String>,
  },
  Show {
    backend: String,
    identifiers: Vec<i64>,
    output: Output,
  },
  Abort {
    backend: String,
    global_flags: GlobalFlags,
    identifiers: Vec<i64>,
  },
  Delete {
    backend: String,
    global_flags: GlobalFlags,
    identifiers: Vec<i64>,
  },
}

impl Workflow {
  pub fn execute(&self, configuration: Configuration) -> Result<()> {
    match self {
      Workflow::List {
        after_date,
        backend,
        before_date,
        name,
        output,
        states,
      } => Self::handle_list_workflows(
        &configuration,
        backend,
        output,
        after_date,
        before_date,
        name,
        states,
      ),
      Workflow::Show {
        backend,
        identifiers,
        output,
      } => Self::handle_show_workflows(&configuration, backend, identifiers, output),

      Workflow::Abort {
        backend,
        global_flags,
        identifiers,
      } => Self::handle_abort_workflows(&configuration, backend, identifiers, global_flags),

      Workflow::Delete {
        backend,
        global_flags,
        identifiers,
      } => Self::handle_delete_workflows(&configuration, backend, identifiers, global_flags),
      _ => unimplemented!(),
    }
  }

  fn handle_list_workflows(
    configuration: &Configuration,
    backend: &str,
    output: &Output,
    after_date: &Option<DateTime<Utc>>,
    before_date: &Option<DateTime<Utc>>,
    name: &Option<String>,
    states: &[String],
  ) -> Result<()> {
    let mut backend_client = configuration.get_backend_client(backend)?;
    let response = backend_client.request(Action::ListWorkflow {
      after_date: *after_date,
      before_date: *before_date,
      name: name.clone(),
      states: Vec::from(states),
    })?;
    if response.status().is_success() {
      print_response::<models::WorkflowMultiple>(response, output)?;
    } else {
      println!(
        "Something went wrong, got response status: {}",
        response.status()
      )
    }
    Ok(())
  }

  fn handle_abort_workflows(
    configuration: &Configuration,
    backend: &str,
    identifiers: &[i64],
    global_flags: &GlobalFlags,
  ) -> Result<()> {
    let string_identifiers = identifiers
      .iter()
      .map(|id| id.to_string())
      .collect::<Vec<String>>()
      .join(", ");

    if global_flags.skip_confirmation
      || get_confirmation(OPERATION_ABORT, "Workflow", &string_identifiers)
    {
      let mut backend_client = configuration.get_backend_client(backend)?;
      for identifier in identifiers.iter() {
        let response = backend_client.request(Action::AbortWorkflow {
          identifier: *identifier,
        })?;
        let message = if response.status().is_success() {
          "Workflow aborted "
        } else {
          "Failed abort workflow "
        };
        println!("{} (n°{})", message, identifier);
      }
    } else {
      println!("Cancelled operation");
    }
    Ok(())
  }

  fn handle_delete_workflows(
    configuration: &Configuration,
    backend: &str,
    identifiers: &[i64],
    global_flags: &GlobalFlags,
  ) -> Result<()> {
    let string_identifiers = identifiers
      .iter()
      .map(|id| id.to_string())
      .collect::<Vec<String>>()
      .join(", ");

    if global_flags.skip_confirmation
      || get_confirmation(OPERATION_DELETE, "Workflow", &string_identifiers)
    {
      let mut backend_client = configuration.get_backend_client(backend)?;
      for identifier in identifiers.iter() {
        let response = backend_client.request(Action::DeleteWorkflow {
          identifier: *identifier,
        })?;
        let message = if response.status().is_success() {
          "Workflow deleted "
        } else {
          "Failed delete workflow "
        };
        println!("{} (n°{})", message, identifier);
      }
    } else {
      println!("Cancelled operation");
    }
    Ok(())
  }

  fn handle_show_workflows(
    configuration: &Configuration,
    backend: &str,
    identifiers: &[i64],
    output: &Output,
  ) -> Result<()> {
    {
      let mut backend_client = configuration.get_backend_client(backend)?;
      for identifier in identifiers.iter() {
        let response = backend_client.request(Action::ShowWorkflow {
          identifier: *identifier,
        })?;
        print_response::<models::WorkflowSingle>(response, output)?;
      }
      Ok(())
    }
  }

  pub fn get_sub_command() -> App<'static, 'static> {
    let backend = Arg::with_name("backend")
      .help("Select the backend configuration")
      .long(PARAMETER_BACKEND)
      .env("BACKEND")
      .default_value("default");

    SubCommand::with_name(MODEL_WORKFLOW)
      .about("Manage workflow")
      .version(built_info::PKG_VERSION)
      .subcommand(
        SubCommand::with_name(OPERATION_DEFINITION)
          .about("Retrieve workflow definitions")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_LIST)
          .about("List workflows")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(super::get_output_parameter())
          .arg(
            Arg::with_name("after_date")
              .short("a")
              .long("after_date")
              .value_name("after_date")
              .help("Sets start date filter")
              .takes_value(true),
          )
          .arg(
            Arg::with_name("before_date")
              .short("b")
              .long("before_date")
              .value_name("before_date")
              .help("Sets end date filter")
              .takes_value(true),
          )
          .arg(
            Arg::with_name("name")
              .short("n")
              .long("name")
              .value_name("name")
              .help("Sets name filter")
              .takes_value(true),
          )
          .arg(
            Arg::with_name("state")
              .short("s")
              .long("state")
              .value_name("state")
              .help("Sets state filter")
              .multiple(true)
              .takes_value(true),
          ),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_SHOW)
          .about("Show a workflow by it's identifier")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Identifier of the workflow")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_ABORT)
          .about("Abort one or more specified workflow")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Identifier of the workflows")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_DELETE)
          .about("Delete one or more specified workflow")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Identifier of the workflows")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_BENCHMARK)
          .about("Benchmark a workflows")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone()),
      )
  }
}

impl<'a> TryFrom<&ArgMatches<'a>> for Workflow {
  type Error = crate::Error;

  fn try_from(matches: &ArgMatches<'a>) -> std::result::Result<Self, Self::Error> {
    match matches.subcommand() {
      (label, parameters) if label == OPERATION_CREATE => {
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Workflow::Create { backend, output })
      }
      (label, parameters) if label == OPERATION_LIST => {
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;
        let name = get_command_option(&parameters, PARAMETER_NAME);

        let optional_after_date = if let Some(date) = get_command_option(&parameters, "after_date")
        {
          parse_date(date)
        } else {
          None
        };

        let optional_before_date =
          if let Some(date) = get_command_option(&parameters, "before_date") {
            parse_date(date)
          } else {
            None
          };

        let optional_states = get_multiple_command_options(&parameters, "state");

        Ok(Workflow::List {
          after_date: optional_after_date,
          backend,
          before_date: optional_before_date,
          name,
          output,
          states: optional_states,
        })
      }
      (label, parameters) if label == OPERATION_SHOW => {
        let identifiers = get_required_parameters_as_integers(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Workflow::Show {
          backend,
          identifiers,
          output,
        })
      }
      (label, parameters) if label == OPERATION_ABORT => {
        let identifiers = get_required_parameters_as_integers(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;

        Ok(Workflow::Abort {
          backend,
          global_flags: GlobalFlags {
            skip_confirmation: false,
          },
          identifiers,
        })
      }
      (label, parameters) if label == OPERATION_DELETE => {
        let identifiers = get_required_parameters_as_integers(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;

        Ok(Workflow::Delete {
          backend,
          global_flags: GlobalFlags {
            skip_confirmation: false,
          },
          identifiers,
        })
      }
      _ => unreachable!(),
    }
  }
}

#[cfg(test)]
mod tests {
  use super::*;
  use crate::{backend::AccessToken, configuration::Backend, models::WorkflowSingle};
  use httpmock::prelude::*;

  #[test]
  fn test_execute() {
    let expected_workflow = WorkflowSingle {
      data: (crate::models::Workflow {
        created_at: "Test-date".to_string(),
        id: 1,
        identifier: "Test_id".to_string(),
        reference: None,
        schema_version: "Test_scheme".to_string(),
        status: None,
        version_major: 9,
        version_minor: 9,
        version_micro: 9,
      }),
    };

    let server = MockServer::start();
    server.mock(|when, then| {
      when.method(GET).path("/api/step_flow/workflows/1");
      then
        .status(200)
        .header("content-type", "application/json")
        .body(serde_json::to_string(&expected_workflow).unwrap());
    });

    let access_token = AccessToken {
      access_token: "".to_string(),
      user: Default::default(),
    };
    server.mock(|when, then| {
      when.method(POST).path("/api/sessions");
      then
        .status(200)
        .header("content-type", "application/json")
        .body(serde_json::to_string(&access_token).unwrap());
    });

    let mut configuration = Configuration::default();
    configuration
      .add_backend(Backend::new(
        "backend_name",
        &server.base_url(),
        "user",
        "password",
      ))
      .unwrap();
    let output = Output::Raw;

    let credential_subcommand = Workflow::Show {
      backend: "backend_name".to_string(),
      identifiers: vec![1],
      output,
    };
    let result = credential_subcommand.execute(configuration);
    assert!(result.is_ok());
  }
}
