use crate::{
  backend::Action,
  configuration::Configuration,
  constant::*,
  error::Result,
  get_confirmation, models, print_response,
  subcommands::{
    get_required_parameter_as_string, get_required_parameters_as_strings, GlobalFlags,
  },
  Output,
};
use clap::{App, Arg, ArgMatches, SubCommand};
use std::convert::TryFrom;

#[derive(Debug)]
pub enum Worker {
  List {
    backend: String,
    output: Output,
  },
  Show {
    backend: String,
    identifiers: Vec<String>,
    output: Output,
  },
  Stop {
    backend: String,
    global_flags: GlobalFlags,
    identifiers: Vec<String>,
    output: Output,
  },
  ConsumingJobResume {
    backend: String,
    global_flags: GlobalFlags,
    identifiers: Vec<String>,
    output: Output,
  },
  ConsumingJobStop {
    backend: String,
    global_flags: GlobalFlags,
    identifiers: Vec<String>,
    output: Output,
  },
}

impl Worker {
  pub fn execute(&self, configuration: Configuration, global_flags: &GlobalFlags) -> Result<()> {
    match self {
      Worker::List { backend, output } => {
        Self::handle_list_workers(&configuration, backend, output)
      }

      Worker::Show {
        backend,
        identifiers,
        output,
      } => Self::handle_show_workers(&configuration, backend, output, identifiers),

      Worker::Stop {
        backend,
        identifiers,
        ..
      } => Self::handle_stop_workers(&configuration, backend, identifiers, global_flags),

      Worker::ConsumingJobStop {
        backend,
        identifiers,
        ..
      } => Self::handle_lock_workers(&configuration, backend, identifiers, global_flags),

      Worker::ConsumingJobResume {
        backend,
        identifiers,
        ..
      } => Self::handle_unlock_workers(&configuration, backend, identifiers, global_flags),
    }
  }

  fn handle_list_workers(
    configuration: &Configuration,
    backend: &str,
    output: &Output,
  ) -> Result<()> {
    let mut backend_client = configuration.get_backend_client(backend)?;
    let response = backend_client.request(Action::ListWorker)?;
    print_response::<models::WorkerMultiple>(response, output)?;
    Ok(())
  }

  fn handle_show_workers(
    configuration: &Configuration,
    backend: &str,
    output: &Output,
    identifiers: &[String],
  ) -> Result<()> {
    let mut backend_client = configuration.get_backend_client(backend)?;
    let _ = identifiers
      .iter()
      .map(|id| {
        let response = backend_client.request(Action::ShowWorker {
          identifier: id.to_string(),
        });
        print_response::<models::WorkerSingle>(response.unwrap(), output)
      })
      .collect::<Result<Vec<()>>>()?;
    Ok(())
  }

  fn handle_lock_workers(
    configuration: &Configuration,
    backend: &str,
    identifiers: &[String],
    global_flags: &GlobalFlags,
  ) -> Result<()> {
    if global_flags.skip_confirmation
      || get_confirmation(
        OPERATION_JOB_CONSUMPTION_STOP,
        "Worker",
        &identifiers.join(", "),
      )
    {
      let mut backend_client = configuration.get_backend_client(backend)?;
      for identifier in identifiers.iter() {
        let response = backend_client.request(Action::WorkerJobConsumptionStop {
          identifier: identifier.clone(),
        })?;
        let message = if response.status().is_success() {
          "Worker job consumption locked (ID {})"
        } else {
          "Failed to lock workers job consumption (ID {})"
        };
        println!("{} (n°{})", message, identifier);
      }
    } else {
      println!("Cancelled operation");
    }
    Ok(())
  }

  fn handle_unlock_workers(
    configuration: &Configuration,
    backend: &str,
    identifiers: &[String],
    global_flags: &GlobalFlags,
  ) -> Result<()> {
    if global_flags.skip_confirmation
      || get_confirmation(
        OPERATION_JOB_CONSUMPTION_RESUME,
        "Worker",
        &identifiers.join(", "),
      )
    {
      let mut backend_client = configuration.get_backend_client(backend)?;
      for identifier in identifiers.iter() {
        let response = backend_client.request(Action::WorkerJobConsumptionResume {
          identifier: identifier.clone(),
        })?;
        let message = if response.status().is_success() {
          "Worker job consumption unlocked (ID {})"
        } else {
          "Failed to unlock workers job consumption (ID {})"
        };
        println!("{} (n°{})", message, identifier);
      }
    } else {
      println!("Cancelled operation");
    }
    Ok(())
  }

  fn handle_stop_workers(
    configuration: &Configuration,
    backend: &str,
    identifiers: &[String],
    global_flags: &GlobalFlags,
  ) -> Result<()> {
    if global_flags.skip_confirmation
      || get_confirmation(OPERATION_STOP, "Worker", &identifiers.join(", "))
    {
      let mut backend_client = configuration.get_backend_client(backend)?;
      for identifier in identifiers.iter() {
        let response = backend_client.request(Action::StopWorker {
          identifier: identifier.clone(),
        })?;
        let message = if response.status().is_success() {
          "Worker job consumption stopped (ID {})"
        } else {
          "Failed to stop workers job consumption (ID {})"
        };
        println!("{} (n°{})", message, identifier);
      }
    } else {
      println!("Cancelled operation");
    }
    Ok(())
  }

  pub fn get_sub_command() -> App<'static, 'static> {
    let backend = Arg::with_name("backend")
      .help("Select the backend configuration")
      .long(PARAMETER_BACKEND)
      .env("BACKEND")
      .default_value("default");

    SubCommand::with_name(MODEL_WORKER)
      .about("Manage worker")
      .version(built_info::PKG_VERSION)
      .subcommand(
        SubCommand::with_name(OPERATION_LIST)
          .about("List workers")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_SHOW)
          .about("Show one or several workers by their identifiers")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Identifier of the worker")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_STOP)
          .about("Stop one or more specified workers")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Identifier of the worker")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_JOB_CONSUMPTION_STOP)
          .about("Order one or more specified workers to stop consuming the job messages.")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Identifier of the workers")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_JOB_CONSUMPTION_RESUME)
          .about("Order one or more specified workers to resume consuming the job messages.")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Identifier of the workers")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_BENCHMARK)
          .about("Benchmark a worker")
          .version(built_info::PKG_VERSION),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_DEFINITION)
          .about("Retrieve workers definitions")
          .version(built_info::PKG_VERSION),
      )
  }
}

impl<'a> TryFrom<&ArgMatches<'a>> for Worker {
  type Error = crate::Error;

  fn try_from(matches: &ArgMatches<'a>) -> std::result::Result<Self, Self::Error> {
    match matches.subcommand() {
      (label, parameters) if label == OPERATION_LIST => {
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Worker::List { backend, output })
      }
      (label, parameters) if label == OPERATION_SHOW => {
        let identifiers = get_required_parameters_as_strings(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Worker::Show {
          backend,
          identifiers,
          output,
        })
      }
      (label, parameters) if label == OPERATION_STOP => {
        let identifiers = get_required_parameters_as_strings(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Worker::Stop {
          backend,
          global_flags: GlobalFlags {
            skip_confirmation: false,
          },
          identifiers,
          output,
        })
      }

      (label, parameters) if label == OPERATION_JOB_CONSUMPTION_STOP => {
        let identifiers = get_required_parameters_as_strings(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Worker::ConsumingJobStop {
          backend,
          global_flags: GlobalFlags {
            skip_confirmation: false,
          },
          identifiers,
          output,
        })
      }
      (label, parameters) if label == OPERATION_JOB_CONSUMPTION_RESUME => {
        let identifiers = get_required_parameters_as_strings(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Worker::ConsumingJobResume {
          backend,
          global_flags: GlobalFlags {
            skip_confirmation: false,
          },
          identifiers,
          output,
        })
      }
      _ => unimplemented!(),
    }
  }
}
