use crate::{
  backend,
  configuration::Configuration,
  constant::*,
  error::Result,
  models, print_response,
  subcommands::{get_required_parameter_as_string, get_required_parameters_as_integers},
  Output,
};
use clap::{App, Arg, ArgMatches, SubCommand};
use std::convert::TryFrom;

#[derive(Debug)]
pub enum User {
  List {
    backend: String,
    output: Output,
  },
  Show {
    backend: String,
    identifiers: Vec<i64>,
    output: Output,
  },
}

impl User {
  pub fn execute(&self, configuration: Configuration) -> Result<()> {
    match self {
      User::List { backend, output } => {
        let mut backend_client = configuration.get_backend_client(backend)?;
        let response = backend_client.request(backend::Action::ListUser)?;
        print_response::<models::UserMultiple>(response, output)?;
        Ok(())
      }
      User::Show {
        backend,
        identifiers,
        output,
      } => {
        let mut backend_client = configuration.get_backend_client(backend)?;
        for identifier in identifiers.iter() {
          let response = backend_client.request(backend::Action::ShowUser {
            identifier: *identifier,
          })?;
          print_response::<models::UserSingle>(response, output)?;
        }
        Ok(())
      }
    }
  }

  pub fn get_sub_command() -> App<'static, 'static> {
    let backend = Arg::with_name("backend")
      .help("Select the backend configuration")
      .long(PARAMETER_BACKEND)
      .env("BACKEND")
      .default_value("default");

    SubCommand::with_name(MODEL_USER)
      .about("Manage users")
      .version(built_info::PKG_VERSION)
      .subcommand(
        SubCommand::with_name(OPERATION_LIST)
          .about("List users")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_SHOW)
          .about("Show a user by it's identifier")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Identifier of the user")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
  }
}

impl<'a> TryFrom<&ArgMatches<'a>> for User {
  type Error = crate::Error;

  fn try_from(matches: &ArgMatches<'a>) -> std::result::Result<Self, Self::Error> {
    match matches.subcommand() {
      (label, parameters) if label == OPERATION_LIST => {
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(User::List { backend, output })
      }
      (label, parameters) if label == OPERATION_SHOW => {
        let identifiers = get_required_parameters_as_integers(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(User::Show {
          backend,
          identifiers,
          output,
        })
      }
      _ => unimplemented!(),
    }
  }
}
