mod backend_configuration;
mod configuration;
mod credential;
mod user;
mod worker;
mod workflow;

pub use backend_configuration::BackendConfiguration;
pub use configuration::Configuration;
pub use credential::Credential;
pub use user::User;
pub use worker::Worker;
pub use workflow::Workflow;

use crate::{constant::*, Error, Result};
use clap::{App, Arg, ArgMatches};
use std::fmt::Display;
use std::{
  convert::TryFrom,
  io::{stdin, stdout, Write},
  str::FromStr,
};

#[derive(Debug)]
pub struct GlobalFlags {
  pub(crate) skip_confirmation: bool,
}

impl GlobalFlags {
  pub(crate) fn new(skip_confirmation: bool) -> Self {
    Self { skip_confirmation }
  }
}

#[derive(Debug)]
pub enum SubCommands {
  Configuration(Configuration),
  Credential(Credential),
  User(User),
  Worker(Worker),
  Workflow(Workflow),
  Unknown,
}

impl SubCommands {
  pub fn get_sub_commands() -> Vec<App<'static, 'static>> {
    vec![
      Configuration::get_sub_command(),
      Credential::get_sub_command(),
      User::get_sub_command(),
      Worker::get_sub_command(),
      Workflow::get_sub_command(),
    ]
  }
}

impl<'a> TryFrom<&clap::ArgMatches<'a>> for SubCommands {
  type Error = crate::Error;

  fn try_from(matches: &clap::ArgMatches<'a>) -> std::result::Result<Self, Self::Error> {
    let sub_command = match matches.subcommand() {
      (label, Some(parameters)) if label == MODEL_CONFIGURATION => {
        SubCommands::Configuration(Configuration::try_from(parameters)?)
      }
      (label, Some(parameters)) if label == MODEL_CREDENTIAL => {
        SubCommands::Credential(Credential::try_from(parameters)?)
      }
      (label, Some(parameters)) if label == MODEL_USER => {
        SubCommands::User(User::try_from(parameters)?)
      }
      (label, Some(parameters)) if label == MODEL_WORKER => {
        SubCommands::Worker(Worker::try_from(parameters)?)
      }
      (label, Some(parameters)) if label == MODEL_WORKFLOW => {
        SubCommands::Workflow(Workflow::try_from(parameters)?)
      }
      _ => SubCommands::Unknown,
    };

    Ok(sub_command)
  }
}

const CONFIRMATION_OK_VALUES: [&str; 2] = ["yes", "y"];

pub fn get_required_parameters_as_strings<'a>(
  matches: &Option<&ArgMatches<'a>>,
  key: &str,
) -> Result<Vec<String>> {
  matches
    .map(|values| values.values_of(key))
    .flatten()
    .ok_or_else(|| Error::MissingParameter("Can't get literals parameters".to_string()))
    .map(|values| values.map(|val| val.to_string()).collect::<Vec<String>>())
}

pub fn get_required_parameter_as_string<'a>(
  matches: &Option<&ArgMatches<'a>>,
  key: &str,
) -> Result<String> {
  let vec_result = get_required_parameters_as_strings(matches, key)?;
  Ok(vec_result[0].clone())
}

fn get_required_parameters_as_integers(
  matches: &Option<&ArgMatches>,
  key: &str,
) -> Result<Vec<i64>> {
  matches
    .map(|values| values.values_of(key))
    .flatten()
    .ok_or_else(|| Error::MissingParameter("Can't get integers parameters".to_string()))
    .map(|values| {
      values
        .map(|value| i64::from_str(value).map_err(Error::ParseInt))
        .collect::<Result<Vec<i64>>>()
    })?
}

pub fn get_required_parameter_as_integer<'a>(
  matches: &Option<&ArgMatches<'a>>,
  key: &str,
) -> Result<i64> {
  let vec_result = get_required_parameters_as_integers(matches, key)?;
  Ok(vec_result[0])
}

pub fn get_output_parameter() -> Arg<'static, 'static> {
  Arg::with_name(PARAMETER_OUTPUT)
    .short("o")
    .possible_values(&["formatted", "raw", "json", "pretty-json"])
    .default_value("formatted")
    .help("Configure the rendering output")
}

pub fn get_confirmation<T: Display>(
  operation: &str,
  selected_object: &str,
  identifier: &T,
) -> bool {
  println!(
    "Are you sure you want to {} {} {} (Yes/No) ?",
    operation, selected_object, identifier
  );

  CONFIRMATION_OK_VALUES.contains(&user_input().to_lowercase().as_str())
}

fn user_input() -> String {
  let mut s = String::new();
  let _ = stdout().flush();
  stdin()
    .read_line(&mut s)
    .expect("Did not enter a correct string");
  if let Some('\n') = s.chars().next_back() {
    s.pop();
  }
  if let Some('\r') = s.chars().next_back() {
    s.pop();
  }
  s
}

pub fn get_command_option<'a>(matches: &Option<&ArgMatches<'a>>, key: &str) -> Option<String> {
  matches
    .map(|matches| matches.value_of(key).map(|v| v.to_owned()))
    .flatten()
}

pub fn get_multiple_command_options<'a>(
  matches: &Option<&ArgMatches<'a>>,
  key: &str,
) -> Vec<String> {
  matches
    .map(|matches| matches.values_of(key))
    .flatten()
    .map(|values| values.map(|value| value.to_string()).collect())
    .unwrap_or_default()
}
