use crate::{
  backend::Action,
  configuration::Configuration,
  constant::*,
  error::Result,
  models::{CredentialContent, CredentialMultiple, CredentialSingle},
  print_response,
  subcommands::{
    get_confirmation, get_required_parameter_as_integer, get_required_parameter_as_string,
    get_required_parameters_as_strings, GlobalFlags,
  },
  Output,
};
use clap::{App, Arg, ArgMatches, SubCommand};
use rpassword::prompt_password_stdout;
use std::convert::TryFrom;

#[derive(Debug)]
pub enum Credential {
  List {
    backend: String,
    output: Output,
  },
  Show {
    backend: String,
    identifiers: Vec<String>,
    output: Output,
  },
  Add {
    backend: String,
    key: String,
    output: Output,
  },
  Delete {
    backend: String,
    global_flags: GlobalFlags,
    identifier: i64,
    output: Output,
  },
}

impl Credential {
  pub fn execute(&self, configuration: Configuration, global_flags: &GlobalFlags) -> Result<()> {
    match self {
      Credential::List { backend, output } => {
        let mut backend_client = configuration.get_backend_client(backend)?;
        let response = backend_client.request(Action::ListCredential)?;
        print_response::<CredentialMultiple>(response, output)?;
        Ok(())
      }
      Credential::Show {
        backend,
        identifiers,
        output,
      } => {
        let mut backend_client = configuration.get_backend_client(backend)?;
        for identifier in identifiers.iter() {
          let response = backend_client.request(Action::ShowCredential {
            identifier: identifier.clone(),
          })?;
          print_response::<CredentialSingle>(response, output)?;
        }
        Ok(())
      }
      Credential::Add { backend, key, .. } => {
        let mut backend_client = configuration.get_backend_client(backend)?;
        let value = prompt_password_stdout("Password: ").unwrap();
        let response = backend_client.request(Action::AddCredential(CredentialContent {
          key: key.to_string(),
          value,
        }))?;
        if response.status().is_success() {
          println!("Credential '{}' added!", key);
        } else {
          println!(
            "Credential '{}' not created, got response status: {}",
            key,
            response.status()
          )
        }
        Ok(())
      }
      Credential::Delete {
        backend,
        identifier,
        ..
      } => {
        if global_flags.skip_confirmation
          || get_confirmation(OPERATION_DELETE, "Credential", &identifier.to_string())
        {
          let mut backend_client = configuration.get_backend_client(backend)?;
          let response = backend_client.request(Action::DeleteCredential {
            identifier: *identifier,
          })?;
          if response.status().is_success() {
            println!("Credential '{}' deleted!", identifier);
          } else {
            println!(
              "Credential '{}' not deleted, got response status: {}",
              identifier,
              response.status()
            )
          }
        } else {
          println!("Cancelled operation");
        }
        Ok(())
      }
    }
  }

  pub fn get_sub_command() -> App<'static, 'static> {
    let backend = Arg::with_name("backend")
      .help("Select the backend configuration")
      .long(PARAMETER_BACKEND)
      .env("BACKEND")
      .default_value("default");

    SubCommand::with_name(MODEL_CREDENTIAL)
      .about("Manage credential")
      .version(built_info::PKG_VERSION)
      .subcommand(
        SubCommand::with_name(OPERATION_LIST)
          .about("List credentials")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_SHOW)
          .about("Show a credential by it's key")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Key of the credential")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .multiple(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_ADD)
          .about("Add a new credential")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("Key of the new credential")
              .long(PARAMETER_NAME)
              .required(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
      .subcommand(
        SubCommand::with_name(OPERATION_DELETE)
          .about("Delete a credential by ID")
          .version(built_info::PKG_VERSION)
          .arg(backend.clone())
          .arg(
            Arg::with_name("identifier")
              .help("ID of the credential to delete")
              .long(PARAMETER_IDENTIFIER)
              .required(true)
              .index(1),
          )
          .arg(super::get_output_parameter()),
      )
  }
}

impl<'a> TryFrom<&ArgMatches<'a>> for Credential {
  type Error = crate::Error;

  fn try_from(matches: &ArgMatches<'a>) -> std::result::Result<Self, Self::Error> {
    match matches.subcommand() {
      (label, parameters) if label == OPERATION_LIST => {
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Credential::List { backend, output })
      }
      (label, parameters) if label == OPERATION_SHOW => {
        let identifiers = get_required_parameters_as_strings(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Credential::Show {
          backend,
          identifiers,
          output,
        })
      }
      (label, parameters) if label == OPERATION_ADD => {
        let key = get_required_parameter_as_string(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Credential::Add {
          backend,
          key,
          output,
        })
      }
      (label, parameters) if label == OPERATION_DELETE => {
        let identifier = get_required_parameter_as_integer(&parameters, PARAMETER_IDENTIFIER)?;
        let backend = get_required_parameter_as_string(&parameters, PARAMETER_BACKEND)?;
        let output = Output::try_from(&parameters)?;

        Ok(Credential::Delete {
          backend,
          output,
          identifier,
          global_flags: GlobalFlags {
            skip_confirmation: false,
          },
        })
      }
      _ => unimplemented!(),
    }
  }
}

#[cfg(test)]
mod tests {
  use super::*;
  use crate::{backend::AccessToken, configuration::Backend, models::CredentialSingle};
  use httpmock::prelude::*;

  #[test]
  fn test_execute() {
    let expected_credential = CredentialSingle {
      data: crate::models::Credential {
        inserted_at: "2021-10-28T12:28:57".to_string(),
        id: 123,
        key: "KEY".to_string(),
        value: "VALUE".to_string(),
      },
    };

    let server = MockServer::start();
    server.mock(|when, then| {
      when.method(GET).path("/api/credentials/KEY");
      then
        .status(200)
        .header("content-type", "application/json")
        .body(serde_json::to_string(&expected_credential).unwrap());
    });

    let access_token = AccessToken {
      access_token: "".to_string(),
      user: Default::default(),
    };
    server.mock(|when, then| {
      when.method(POST).path("/api/sessions");
      then
        .status(200)
        .header("content-type", "application/json")
        .body(serde_json::to_string(&access_token).unwrap());
    });

    let mut configuration = Configuration::default();
    configuration
      .add_backend(Backend::new(
        "backend_name",
        &server.base_url(),
        "user",
        "password",
      ))
      .unwrap();
    let output = Output::Raw;

    let credential_subcommand = Credential::Show {
      backend: "backend_name".to_string(),
      identifiers: vec!["KEY".to_string()],
      output,
    };
    let global_flag = GlobalFlags::new(true);
    let result = credential_subcommand.execute(configuration, &global_flag);
    assert!(result.is_ok());
  }
}
