use super::BackendConfiguration;
use crate::constant::*;
use clap::{App, Arg, ArgMatches, SubCommand};
use std::convert::TryFrom;

#[derive(Debug)]
pub enum Configuration {
  Backend(BackendConfiguration),
}

impl Configuration {
  pub fn get_sub_command() -> App<'static, 'static> {
    SubCommand::with_name(MODEL_CONFIGURATION)
      .about("Configure credentials (Backend, Docker registries, etc.)")
      .version(built_info::PKG_VERSION)
      .subcommand(
        SubCommand::with_name(CONFIGURATION_BACKEND)
          .about("Configure backend access")
          .version(built_info::PKG_VERSION)
          .subcommand(Self::get_add_backend_sub_command())
          .subcommand(Self::get_delete_backend_sub_command())
          .subcommand(Self::get_list_backend_sub_command())
          .subcommand(Self::get_show_backend_sub_command()),
      )
      .subcommand(
        SubCommand::with_name(CONFIGURATION_REGISTRY)
          .about("Configure Docker registry")
          .version(built_info::PKG_VERSION),
      )
      .subcommand(
        SubCommand::with_name(CONFIGURATION_WORKER)
          .about("Configure Worker source code access")
          .version(built_info::PKG_VERSION),
      )
      .subcommand(
        SubCommand::with_name(CONFIGURATION_IMPORT)
          .about("Import configuration")
          .version(built_info::PKG_VERSION),
      )
      .subcommand(
        SubCommand::with_name(CONFIGURATION_EXPORT)
          .about("Export configuration")
          .version(built_info::PKG_VERSION),
      )
  }

  fn get_add_backend_sub_command() -> App<'static, 'static> {
    SubCommand::with_name(OPERATION_ADD)
      .about("Register a new backend configuration")
      .arg(
        Arg::with_name("name")
          .help("Identifier for this configuration")
          .long(PARAMETER_NAME)
          .default_value("default"),
      )
      .arg(
        Arg::with_name("hostname")
          .help("Hostname of the backend, example: https://backend.mcai.com")
          .long(PARAMETER_HOSTNAME)
          .takes_value(true)
          .required(true),
      )
      .arg(
        Arg::with_name("username")
          .help("Username to login")
          .long(PARAMETER_USERNAME)
          .takes_value(true)
          .required(true),
      )
      .arg(
        Arg::with_name("password")
          .help("Password to login")
          .long(PARAMETER_PASSWORD)
          .takes_value(true)
          .required(true),
      )
  }

  fn get_delete_backend_sub_command() -> App<'static, 'static> {
    SubCommand::with_name(OPERATION_DELETE)
      .about("Delete a specific backend configuration")
      .arg(
        Arg::with_name("name")
          .help("Identifier for this configuration")
          .long(PARAMETER_NAME)
          .default_value("default")
          .index(1),
      )
  }

  fn get_list_backend_sub_command() -> App<'static, 'static> {
    SubCommand::with_name(OPERATION_LIST).about("List backend configurations")
  }

  fn get_show_backend_sub_command() -> App<'static, 'static> {
    SubCommand::with_name(OPERATION_SHOW)
      .about("Show a specific backend configuration")
      .arg(
        Arg::with_name("name")
          .help("Identifier for this configuration")
          .long(PARAMETER_NAME)
          .default_value("default")
          .index(1),
      )
  }
}

impl<'a> TryFrom<&ArgMatches<'a>> for Configuration {
  type Error = crate::Error;

  fn try_from(matches: &ArgMatches<'a>) -> std::result::Result<Self, Self::Error> {
    match matches.subcommand() {
      (label, Some(parameters)) if label == CONFIGURATION_BACKEND => Ok(Configuration::Backend(
        BackendConfiguration::try_from(parameters)?,
      )),
      _ => unreachable!(),
    }
  }
}
