use super::get_required_parameter_as_string;
use crate::constant::*;
use clap::ArgMatches;
use std::convert::TryFrom;

#[derive(Debug)]
pub enum BackendConfiguration {
  Add {
    hostname: String,
    name: String,
    password: String,
    username: String,
  },
  Delete(String),
  List,
  Show(String),
}

impl<'a> TryFrom<&ArgMatches<'a>> for BackendConfiguration {
  type Error = crate::Error;

  fn try_from(matches: &ArgMatches<'a>) -> std::result::Result<Self, Self::Error> {
    match matches.subcommand() {
      (label, parameters) if label == OPERATION_ADD => {
        let hostname = get_required_parameter_as_string(&parameters, PARAMETER_HOSTNAME)?;
        let name = get_required_parameter_as_string(&parameters, PARAMETER_NAME)?;
        let password = get_required_parameter_as_string(&parameters, PARAMETER_PASSWORD)?;
        let username = get_required_parameter_as_string(&parameters, PARAMETER_USERNAME)?;

        Ok(BackendConfiguration::Add {
          hostname,
          name,
          password,
          username,
        })
      }
      (label, parameters) if label == OPERATION_DELETE => Ok(BackendConfiguration::Delete(
        get_required_parameter_as_string(&parameters, PARAMETER_NAME)?,
      )),
      (label, Some(_)) if label == OPERATION_LIST => Ok(BackendConfiguration::List),
      (label, parameters) if label == OPERATION_SHOW => Ok(BackendConfiguration::Show(
        get_required_parameter_as_string(&parameters, PARAMETER_NAME)?,
      )),
      _ => unreachable!(),
    }
  }
}
