use serde::{Deserialize, Serialize};
use term_table::{row::Row, table_cell::TableCell};

#[derive(Deserialize, Serialize, PartialEq, Debug)]
pub struct Workflow {
  pub(crate) created_at: String,
  pub(crate) id: u32,
  pub(crate) identifier: String,
  pub(crate) reference: Option<String>,
  pub(crate) schema_version: String,
  pub(crate) status: Option<Status>,
  pub(crate) version_major: u32,
  pub(crate) version_minor: u32,
  pub(crate) version_micro: u32,
}

#[derive(Deserialize, Serialize, PartialEq, Debug, Clone)]
pub struct Status {
  pub id: u32,
  pub inserted_at: String,
  pub state: String,
}

impl<'a> From<&Workflow> for Row<'a> {
  fn from(workflow: &Workflow) -> Row<'a> {
    Row::new(vec![
      TableCell::new(format!("{}", workflow.id)),
      TableCell::new(&workflow.created_at),
      TableCell::new(&workflow.identifier),
      TableCell::new(format!(
        "{}.{}.{}",
        workflow.version_major, workflow.version_minor, workflow.version_micro
      )),
      TableCell::new(&workflow.schema_version),
      TableCell::new(
        workflow
          .status
          .as_ref()
          .map(|status| status.state.clone())
          .unwrap_or_else(|| "-".to_string()),
      ),
      TableCell::new(workflow.reference.clone().unwrap_or_default()),
    ])
  }
}

pub fn workflow_columns<'a>() -> Row<'a> {
  Row::new(vec![
    TableCell::new("ID"),
    TableCell::new("Created at"),
    TableCell::new("Identifier"),
    TableCell::new("Version"),
    TableCell::new("Schema Version"),
    TableCell::new("State"),
    TableCell::new("Reference"),
  ])
}

#[cfg(test)]
mod test {
  use crate::models::workflows::workflow::Status;
  use crate::models::Workflow;
  use term_table::{row::Row, table_cell::TableCell};

  #[test]
  fn from_workflow_to_row() {
    let workflow = Workflow {
      created_at: "test_date".to_string(),
      id: 1,
      identifier: "test_id".to_string(),
      reference: None,
      schema_version: "test_schema_version".to_string(),
      status: Option::from(Status {
        id: 0,
        inserted_at: "".to_string(),
        state: "pending".to_string(),
      }),
      version_major: 1,
      version_minor: 1,
      version_micro: 1,
    };

    let row = Row {
      cells: vec![
        TableCell::new("1"),
        TableCell::new("test_date"),
        TableCell::new("test_id"),
        TableCell::new("1.1.1"),
        TableCell::new("test_schema_version"),
        TableCell::new("pending"),
        TableCell::new(""),
      ],
      has_separator: true,
    };

    let result = Row::from(&workflow);
    let row_string = format!("{:?}", row);
    let result_string = format!("{:?}", result);

    assert_eq!(result_string, row_string);
  }
}
