use crate::constant::HIDDEN_VALUE;
use serde::{Deserialize, Serialize};
use term_table::{row::Row, table_cell::TableCell};

#[derive(Serialize)]
pub struct CredentialContent {
  pub(crate) key: String,
  pub(crate) value: String,
}

#[derive(Deserialize, Serialize)]
pub struct Credential {
  pub(crate) inserted_at: String,
  pub(crate) id: i32,
  pub(crate) key: String,
  pub(crate) value: String,
}

impl<'a> From<&Credential> for Row<'a> {
  fn from(credential: &Credential) -> Row<'a> {
    Row::new(vec![
      TableCell::new(format!("{}", credential.id)),
      TableCell::new(&credential.inserted_at),
      TableCell::new(credential.key.to_string()),
      TableCell::new(credential.value.to_string()),
    ])
  }
}

impl Credential {
  pub fn create_row_with_hidden_value(&self) -> Row {
    Row::new(vec![
      TableCell::new(format!("{}", self.id)),
      TableCell::new(&self.inserted_at),
      TableCell::new(self.key.to_string()),
      TableCell::new(HIDDEN_VALUE.to_string()),
    ])
  }
}

pub fn credential_columns<'a>() -> Row<'a> {
  Row::new(vec![
    TableCell::new("ID"),
    TableCell::new("Created at"),
    TableCell::new("Key"),
    TableCell::new("Value"),
  ])
}

#[cfg(test)]
mod test {
  use super::*;
  use term_table::{row::Row, table_cell::TableCell};

  #[test]
  fn from_credential_to_row() {
    let credential = Credential {
      inserted_at: "test_date".to_string(),
      id: 1,
      key: "test_key".to_string(),
      value: "test_value".to_string(),
    };

    let row = Row {
      cells: vec![
        TableCell::new("1"),
        TableCell::new("test_date"),
        TableCell::new("test_key"),
        TableCell::new("test_value"),
      ],
      has_separator: true,
    };

    let result = Row::from(&credential);
    let row_string = format!("{:?}", row);
    let result_string = format!("{:?}", result);
    assert_eq!(result_string, row_string);
  }
}
