mod backend;
mod configuration;
mod constant;
mod datetime;
mod error;
mod models;
mod output;
mod subcommands;

use crate::subcommands::{
  get_confirmation, BackendConfiguration, Configuration::Backend, GlobalFlags, SubCommands,
};
use clap::{App, Arg};
use constant::*;
use env_logger::Builder;
use error::{Error, Result};
use output::Output;
use std::{convert::TryFrom, io::Write, process::exit};

fn main() -> Result<()> {
  let matches = App::new(built_info::PKG_NAME)
    .about("Interact with Media Cloud AI platform")
    .version(built_info::PKG_VERSION)
    .arg(
      Arg::with_name("skip_confirmation")
        .short("y")
        .global(true)
        .help("Disable confirmation message"),
    )
    .subcommands(SubCommands::get_sub_commands())
    .get_matches();

  let mut builder = Builder::from_default_env();
  builder
    .format(move |stream, record| writeln!(stream, "[{}] {}", record.level(), record.args(),))
    .init();

  let sub_command = SubCommands::try_from(&matches)?;

  let global_flags = GlobalFlags::new(matches.is_present("skip_confirmation"));

  log::debug!("{:?}", sub_command);

  if let Err(error) = process(&sub_command, global_flags) {
    println!("ERROR: {}", error);
    exit(-1);
  }

  Ok(())
}

fn process(sub_command: &SubCommands, global_flags: GlobalFlags) -> Result<()> {
  let mut configuration = configuration::Configuration::load()?;

  match sub_command {
    //TODO : Refactor subcommands Configuration -> create execute()
    SubCommands::Configuration(Backend(BackendConfiguration::Add {
      hostname,
      name,
      password,
      username,
    })) => {
      let backend = configuration::Backend::new(name, hostname, username, password);
      configuration.add_backend(backend)?;
      configuration.save()?;
    }

    SubCommands::Configuration(Backend(BackendConfiguration::Delete(identifier))) => {
      if get_confirmation(OPERATION_DELETE, "Backend", identifier) {
        configuration.delete_backend(identifier)?;
        configuration.save()?;
      } else {
        println!("Cancelled operation");
      }
    }
    SubCommands::Configuration(Backend(BackendConfiguration::List)) => {
      configuration.list_backends();
    }
    SubCommands::Configuration(Backend(BackendConfiguration::Show(identifier))) => {
      configuration.show_backend(identifier)?;
    }

    SubCommands::Credential(credential) => credential.execute(configuration, &global_flags)?,

    SubCommands::User(user) => user.execute(configuration)?,

    SubCommands::Workflow(workflow) => workflow.execute(configuration)?,

    SubCommands::Worker(worker) => worker.execute(configuration, &global_flags)?,

    _ => unimplemented!(),
  };

  Ok(())
}

fn print_response<T>(response: reqwest::blocking::Response, output: &Output) -> Result<()>
where
  T: for<'de> serde::Deserialize<'de> + models::Render,
{
  match output {
    Output::Formatted => {
      let parsed_items: T = response.json()?;
      parsed_items.render();
    }
    Output::Json(false) => {
      let content: serde_json::Value = response.json()?;
      println!("{}", serde_json::to_string(&content)?);
    }
    Output::Json(true) => {
      let content: serde_json::Value = response.json()?;
      println!("{}", serde_json::to_string_pretty(&content)?);
    }
    Output::Raw => {
      println!("{}", response.text()?);
    }
  }
  Ok(())
}
