use std::fmt::{self, Display, Formatter};

pub type Result<T> = std::result::Result<T, Error>;

#[derive(Debug)]
pub enum Error {
  BackendNameAlreadyTaken(String),
  Confy(confy::ConfyError),
  MissingParameter(String),
  NoBackendWithName(String),
  ParseInt(std::num::ParseIntError),
  Reqwest(reqwest::Error),
  SerdeJson(serde_json::Error),
}

impl std::error::Error for Error {}

impl From<confy::ConfyError> for Error {
  fn from(error: confy::ConfyError) -> Self {
    Error::Confy(error)
  }
}

impl From<reqwest::Error> for Error {
  fn from(error: reqwest::Error) -> Self {
    Error::Reqwest(error)
  }
}

impl From<serde_json::Error> for Error {
  fn from(error: serde_json::Error) -> Self {
    Error::SerdeJson(error)
  }
}

impl From<std::num::ParseIntError> for Error {
  fn from(error: std::num::ParseIntError) -> Self {
    Error::ParseInt(error)
  }
}

impl Display for Error {
  fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
    match self {
      Error::BackendNameAlreadyTaken(identifier) => {
        write!(f, "Backend name is already taken: {}", identifier)
      }
      Error::Confy(error) => write!(f, "Configuration error: {}", error),
      Error::MissingParameter(parameter) => write!(f, "Missing parameter: {}", parameter),
      Error::NoBackendWithName(name) => write!(f, "No backend with name: {}", name),
      Error::ParseInt(error) => write!(f, "Unable to parse number: {}", error),
      Error::Reqwest(error) => write!(f, "HTTP request: {}", error),
      Error::SerdeJson(error) => write!(f, "JSON: {}", error),
    }
  }
}
