use crate::{backend::action::WorkflowAction, models::CredentialContent};
pub use action::WorkerAction;
use chrono::{DateTime, Utc};
pub use client::{AccessToken, Client, Login};
use reqwest::Method;

mod action;
mod client;

pub enum Action {
  AbortWorkflow {
    identifier: i64,
  },
  AddCredential(CredentialContent),
  DeleteCredential {
    identifier: i64,
  },
  DeleteWorkflow {
    identifier: i64,
  },
  ListCredential,
  ListUser,
  ListWorker,
  ListWorkflow {
    after_date: Option<DateTime<Utc>>,
    before_date: Option<DateTime<Utc>>,
    name: Option<String>,
    states: Vec<String>,
  },
  ShowCredential {
    identifier: String,
  },
  ShowUser {
    identifier: i64,
  },
  ShowWorker {
    identifier: String,
  },
  ShowWorkflow {
    identifier: i64,
  },
  StopWorker {
    identifier: String,
  },
  WorkerJobConsumptionResume {
    identifier: String,
  },
  WorkerJobConsumptionStop {
    identifier: String,
  },
}

impl Action {
  pub fn get_url(&self) -> String {
    let endpoint = match self {
      Action::AbortWorkflow { identifier } | Action::DeleteWorkflow { identifier } => {
        format!("/step_flow/workflows/{}/events", identifier)
      }
      Action::AddCredential(_) | Action::ListCredential => "/credentials".to_string(),
      Action::DeleteCredential { identifier } => {
        format!("/credentials/{}", identifier)
      }
      Action::ListUser => "/users".to_string(),
      Action::ListWorker => "/step_flow/workers".to_string(),
      Action::ListWorkflow {
        name,
        after_date,
        before_date,
        states,
      } => {
        let base_url = "/step_flow/workflows".to_string();
        let mut url_options = vec![];
        if let Some(id) = name {
          url_options.push(format!("workflow_ids[]={}", id))
        }
        if let Some(date) = after_date {
          url_options.push(format!("after_date={}", date.format("%Y-%m-%dT%H:%M:%S")))
        }
        if let Some(date) = before_date {
          url_options.push(format!("before_date={}", date.format("%Y-%m-%dT%H:%M:%S")))
        }
        for state in states {
          url_options.push(format!("states[]={}", state))
        }
        format!("{}?{}", base_url, url_options.join("&"))
      }

      Action::ShowCredential { identifier } => format!("/credentials/{}", identifier),
      Action::ShowUser { identifier } => format!("/users/{}", identifier),
      Action::ShowWorkflow { identifier } => {
        format!("/step_flow/workflows/{}?mode=simple", identifier)
      }
      Action::ShowWorker { identifier }
      | Action::StopWorker { identifier }
      | Action::WorkerJobConsumptionResume { identifier }
      | Action::WorkerJobConsumptionStop { identifier } => {
        format!("/step_flow/workers/{}", identifier)
      }
    };
    format!("/api{}", endpoint)
  }

  pub fn get_body(&self) -> Option<String> {
    match self {
      Action::AbortWorkflow { .. } => WorkflowAction::Abort.as_body().ok(),
      Action::AddCredential(credential) => serde_json::to_string(credential).ok(),
      Action::DeleteWorkflow { .. } => WorkflowAction::Delete.as_body().ok(),
      Action::StopWorker { .. } => WorkerAction::StopWorker.as_body().ok(),
      Action::WorkerJobConsumptionResume { .. } => WorkerAction::ResumeConsumingJobs.as_body().ok(),
      Action::WorkerJobConsumptionStop { .. } => WorkerAction::StopConsumingJobs.as_body().ok(),
      _ => None,
    }
  }

  pub fn get_method(&self) -> Method {
    match self {
      Action::AbortWorkflow { .. }
      | Action::AddCredential { .. }
      | Action::DeleteWorkflow { .. } => Method::POST,
      Action::DeleteCredential { .. } => Method::DELETE,
      Action::ListCredential
      | Action::ListUser
      | Action::ListWorker
      | Action::ListWorkflow { .. }
      | Action::ShowCredential { .. }
      | Action::ShowUser { .. }
      | Action::ShowWorker { .. }
      | Action::ShowWorkflow { .. } => Method::GET,
      Action::StopWorker { .. }
      | Action::WorkerJobConsumptionResume { .. }
      | Action::WorkerJobConsumptionStop { .. } => Method::PUT,
    }
  }

  pub fn get_content_type(&self) -> Option<String> {
    match self {
      Action::AbortWorkflow { .. }
      | Action::AddCredential { .. }
      | Action::DeleteWorkflow { .. }
      | Action::StopWorker { .. }
      | Action::WorkerJobConsumptionResume { .. }
      | Action::WorkerJobConsumptionStop { .. } => Some("application/json".to_string()),
      _ => None,
    }
  }
}
