mod access_token;
mod login;
mod session;
mod user;

use super::Action;
use crate::{configuration::Backend as BackendConfiguration, error::Error, Result};
pub use access_token::AccessToken;
pub use login::Login;
use reqwest::{
  blocking::{Client as ReqwestClient, Response},
  header::CONTENT_TYPE,
};
use session::Session;
use user::User;

pub struct Client {
  client: ReqwestClient,
  token: Option<String>,
  configuration: BackendConfiguration,
}

impl Client {
  pub fn new(configuration: BackendConfiguration) -> Result<Self> {
    let client = ReqwestClient::new();

    Ok(Client {
      client,
      token: None,
      configuration,
    })
  }

  fn get_authentication_token(&mut self) -> Result<String> {
    if let Some(token) = &self.token {
      return Ok(token.clone());
    }

    let login_body = Login::from(&self.configuration).to_string();
    let url = self.configuration.get_session_url();
    log::debug!("Session URL: {}", url);

    let token_response: AccessToken = self
      .client
      .post(url)
      .header(CONTENT_TYPE, "application/json")
      .body(login_body)
      .send()?
      .json()?;

    self.token = Some(token_response.access_token.clone());
    Ok(token_response.access_token)
  }

  pub fn request(&mut self, action: Action) -> Result<Response> {
    let url = self.configuration.build_url(&action.get_url());

    let request = self
      .client
      .request(action.get_method(), url)
      .bearer_auth(self.get_authentication_token()?);

    let request = if let Some(body) = action.get_body() {
      request
        .body(body)
        .header(CONTENT_TYPE, action.get_content_type().unwrap())
    } else {
      request
    };
    request.send().map_err(Error::Reqwest)
  }
}
