use crate::error::Result;
use chrono::{DateTime, Utc};
use mcai_models::CredentialContent;
use reqwest::Method;
use serde::Serialize;

pub enum Action {
  AddCredential(CredentialContent),
  DeleteCredential {
    identifier: i64,
  },
  GetWorkflowDuration {
    identifier: i64,
  },
  ListCredential,
  ListUser,
  ListWorker,
  ListWorkflow {
    after_date: Option<DateTime<Utc>>,
    before_date: Option<DateTime<Utc>>,
    name: Option<String>,
    states: Vec<String>,
  },
  ShowCredential {
    identifier: String,
  },
  ShowUser {
    identifier: i64,
  },
  ShowWorker {
    identifier: String,
  },
  ShowWorkflow {
    identifier: i64,
    mode: ShowWorkflowMode,
  },
  StopWorker {
    identifier: String,
  },
  WorkerJobConsumptionResume {
    identifier: String,
  },
  WorkerJobConsumptionStop {
    identifier: String,
  },
}

pub enum ShowWorkflowMode {
  Full,
  Simple,
}

impl Action {
  pub fn get_url(&self) -> String {
    let endpoint = match self {
      Action::AddCredential(_) | Action::ListCredential => "/credentials".to_string(),
      Action::DeleteCredential { identifier } => {
        format!("/credentials/{}", identifier)
      }
      Action::GetWorkflowDuration { identifier } => {
        format!("/step_flow/durations/workflows?workflow_id={}", identifier)
      }
      Action::ListUser => "/users".to_string(),
      Action::ListWorker => "/step_flow/workers".to_string(),
      Action::ListWorkflow {
        name,
        after_date,
        before_date,
        states,
      } => {
        let base_url = "/step_flow/workflows".to_string();
        let mut url_options = vec![];
        if let Some(id) = name {
          url_options.push(format!("workflow_ids[]={}", id))
        }
        if let Some(date) = after_date {
          url_options.push(format!("after_date={}", date.format("%Y-%m-%dT%H:%M:%S")))
        }
        if let Some(date) = before_date {
          url_options.push(format!("before_date={}", date.format("%Y-%m-%dT%H:%M:%S")))
        }
        for state in states {
          url_options.push(format!("states[]={}", state))
        }
        format!("{}?{}", base_url, url_options.join("&"))
      }

      Action::ShowCredential { identifier } => format!("/credentials/{}", identifier),
      Action::ShowUser { identifier } => format!("/users/{}", identifier),
      Action::ShowWorkflow { identifier, mode } => match mode {
        ShowWorkflowMode::Full => format!("/step_flow/workflows/{}", identifier),
        ShowWorkflowMode::Simple => format!("/step_flow/workflows/{}?mode=simple", identifier),
      },
      Action::ShowWorker { identifier }
      | Action::StopWorker { identifier }
      | Action::WorkerJobConsumptionResume { identifier }
      | Action::WorkerJobConsumptionStop { identifier } => {
        format!("/step_flow/workers/{}", identifier)
      }
    };
    format!("/api{}", endpoint)
  }

  pub fn get_body(&self) -> Option<String> {
    match self {
      Action::AddCredential(credential) => serde_json::to_string(credential).ok(),
      Action::StopWorker { .. } => WorkerAction::StopWorker.as_body().ok(),
      Action::WorkerJobConsumptionResume { .. } => WorkerAction::ResumeConsumingJobs.as_body().ok(),
      Action::WorkerJobConsumptionStop { .. } => WorkerAction::StopConsumingJobs.as_body().ok(),
      _ => None,
    }
  }

  pub fn get_method(&self) -> Method {
    match self {
      Action::AddCredential { .. } => Method::POST,
      Action::DeleteCredential { .. } => Method::DELETE,
      Action::GetWorkflowDuration { .. }
      | Action::ListCredential
      | Action::ListUser
      | Action::ListWorker
      | Action::ListWorkflow { .. }
      | Action::ShowCredential { .. }
      | Action::ShowUser { .. }
      | Action::ShowWorker { .. }
      | Action::ShowWorkflow { .. } => Method::GET,
      Action::StopWorker { .. }
      | Action::WorkerJobConsumptionResume { .. }
      | Action::WorkerJobConsumptionStop { .. } => Method::PUT,
    }
  }

  pub fn get_content_type(&self) -> Option<String> {
    match self {
      Action::AddCredential { .. }
      | Action::StopWorker { .. }
      | Action::WorkerJobConsumptionResume { .. }
      | Action::WorkerJobConsumptionStop { .. } => Some("application/json".to_string()),
      _ => None,
    }
  }
}

#[derive(Serialize)]
#[serde(tag = "type", rename_all = "snake_case")]
pub enum WorkerAction {
  ResumeConsumingJobs,
  StopConsumingJobs,
  StopWorker,
}

impl WorkerAction {
  pub(crate) fn as_body(&self) -> Result<String> {
    Ok(serde_json::to_string(self)?)
  }
}
