use mc_core::world::level::{LevelEnv, Level};
use mc_core::world::chunk::{Chunk, SubChunk, ChunkHeight};
use mc_core::world::anvil::source::{AnvilLevelSource};
use mc_vanilla::ext::WithVanilla;

use std::mem::size_of;
use std::sync::Arc;
use std::time::Duration;


/// This example can be run to test which block states or properties are not properly decoded
/// from a debug world. This example currently supports 1.17.1 and before. Debug worlds in 1.18+
/// requires a different chunk height so you can change it in the function to debug 1.18 block
/// states, even if they are identical to 1.17.1.
///
/// Put the path to the directory of the world into the 'MCRS_LEVEL_DIR' environment variable.
fn main() {

    println!("===== MEMORY USAGE =====");
    let chunk_sizeof = size_of::<Chunk>();
    let sub_chunk_sizeof = size_of::<SubChunk>();
    println!("Chunk height sizeof: {}", size_of::<ChunkHeight>());
    println!("Level sizeof: {}", size_of::<Level>());
    println!("Chunk sizeof: {}", chunk_sizeof);
    println!("SubChunk sizeof: {}", sub_chunk_sizeof);
    println!("For a whole loaded region: {}", 32 * 32 * (chunk_sizeof + 16 * sub_chunk_sizeof));
    println!("========================");
    println!();

    println!("====== ANVIL TEST ======");

    let level_dir = std::env::var("MCRS_LEVEL_DIR").unwrap();
    let env = Arc::new(LevelEnv::with_vanilla());
    let source = AnvilLevelSource::new(level_dir);
    let height = ChunkHeight {
        min: 0,
        max: 15
    };

    let mut level = Level::new("overworld".to_string(), env, height, source);

    println!("Level height: {:?}", level.get_height());

    for cx in 0..=17 {
        for cz in 0..=17 {
            level.request_chunk_load(cx, cz);
        }
    }

    loop {
        level.load_chunks();
        std::thread::sleep(Duration::from_secs(1));
    }

}
