use std::error;
use std::fmt;

#[derive(Debug, Clone)]
pub struct ResponseContent<T> {
    pub status: reqwest::StatusCode,
    pub content: String,
    pub entity: Option<T>,
}

#[derive(Debug)]
pub enum Error<T> {
    Reqwest(reqwest::Error),
    Serde(serde_json::Error),
    Io(std::io::Error),
    ResponseError(ResponseContent<T>),
}

impl <T> fmt::Display for Error<T> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let (module, e) = match self {
            Error::Reqwest(e) => ("reqwest", e.to_string()),
            Error::Serde(e) => ("serde", e.to_string()),
            Error::Io(e) => ("IO", e.to_string()),
            Error::ResponseError(e) => ("response", format!("status code {}", e.status)),
        };
        write!(f, "error in {}: {}", module, e)
    }
}

impl <T: fmt::Debug> error::Error for Error<T> {
    fn source(&self) -> Option<&(dyn error::Error + 'static)> {
        Some(match self {
            Error::Reqwest(e) => e,
            Error::Serde(e) => e,
            Error::Io(e) => e,
            Error::ResponseError(_) => return None,
        })
    }
}

impl <T> From<reqwest::Error> for Error<T> {
    fn from(e: reqwest::Error) -> Self {
        Error::Reqwest(e)
    }
}

impl <T> From<serde_json::Error> for Error<T> {
    fn from(e: serde_json::Error) -> Self {
        Error::Serde(e)
    }
}

impl <T> From<std::io::Error> for Error<T> {
    fn from(e: std::io::Error) -> Self {
        Error::Io(e)
    }
}

pub fn urlencode<T: AsRef<str>>(s: T) -> String {
    ::url::form_urlencoded::byte_serialize(s.as_ref().as_bytes()).collect()
}

pub mod authentication_api;
pub mod bleve_api;
pub mod bots_api;
pub mod brand_api;
pub mod channels_api;
pub mod cloud_api;
pub mod cluster_api;
pub mod commands_api;
pub mod compliance_api;
pub mod data_retention_api;
pub mod elasticsearch_api;
pub mod emoji_api;
pub mod exports_api;
pub mod files_api;
pub mod groups_api;
pub mod imports_api;
pub mod integration_actions_api;
pub mod jobs_api;
pub mod ldap_api;
pub mod migrate_api;
pub mod o_auth_api;
pub mod open_graph_api;
pub mod permissions_api;
pub mod plugins_api;
pub mod posts_api;
pub mod preferences_api;
pub mod reactions_api;
pub mod roles_api;
pub mod root_api;
pub mod saml_api;
pub mod schemes_api;
pub mod search_api;
pub mod shared_channels_api;
pub mod status_api;
pub mod system_api;
pub mod teams_api;
pub mod terms_of_service_api;
pub mod threads_api;
pub mod uploads_api;
pub mod users_api;
pub mod webhooks_api;

pub mod configuration;
