//! ## Deprecation Notice
//!
//! Ruma has support for parsing both types of Matrix URIs as of version
//! `0.5.0`, thus this crate is now deprecated.
//!
//! ## Original Description
//!
//! A crate for building and parsing Matrix URIs according to both the [matrix.to specification](https://matrix.org/docs/spec/appendices#matrix-to-navigation) and also [MSC2312](https://github.com/matrix-org/matrix-doc/pull/2312) (`matrix://`).
//!
//! Optionally provides conversion to types from [`ruma_identifiers`](https://docs.rs/ruma-identifiers/0.17.4/ruma_identifiers/) with the `ruma` feature
//!
//! ## Usage
//!
//! ### Building
//!
//! ```rust
//! use matrix_uri::{MatrixUri, MatrixId, MatrixUriAction, IdType};
//! let uri = MatrixUri::new(
//!             MatrixId::new(IdType::UserId, String::from("cute:some.url")),
//!             None,
//!             None,
//!             Some(vec![String::from("headpat.services"), String::from("cute.local")]),
//!             Some(MatrixUriAction::Chat),
//!         )
//!         .unwrap();
//!
//! println!("{} with {}", uri.action().unwrap().to_string(), uri.mxid.to_string()); // `chat with @cute:some.url`
//! println!("{}", uri.matrix_uri_string()); // `matrix:u/cute%3Asome.url?action=chat&via=headpat.services&via=cute.local`
//! println!("{}", uri.matrix_to_string()); // `https://matrix.to/#/%40cute%3Asome.url?action=chat&via=headpat.services&via=cute.local`
//! ```
//!
//! ### Parsing
//!
//! ```rust
//! use matrix_uri::MatrixUri;
//! let uri = "matrix:u/her:example.org?action=chat".parse::<MatrixUri>().unwrap();
//!
//! println!("{} - {:?}", uri.mxid.to_string(), uri.action().unwrap()); // `@her:example.org - Chat`
//! ```

#![deny(
	missing_docs,
	trivial_casts,
	trivial_numeric_casts,
	unused_extern_crates,
	unused_import_braces,
	unused_qualifications
)]
#![warn(missing_debug_implementations, dead_code, clippy::unwrap_used, clippy::expect_used)]

#[macro_use]
extern crate lazy_static;

#[cfg(feature = "ruma")]
use std::convert::{TryFrom, TryInto};
use std::{collections::HashMap, fmt::Display, str::FromStr};

use regex::{Captures, Regex};
use url::Url;

/// MXID types, documentation can be found [here](https://matrix.org/docs/spec/appendices#common-identifier-format)
#[derive(Clone, PartialEq, Eq, Debug)]
pub enum IdType {
	/// Identifies a user
	UserId,
	/// Identifies a room
	RoomId,
	/// Identifies an event
	EventId,
	/// Identifies a group
	GroupId,
	/// Identifies a room via an alias in the form `#room_alias:domain`
	RoomAlias,
	/// Wraps unknown ID type
	Unknown(String),
}

impl IdType {
	/// Returns the type encoded as a sigil
	pub fn to_sigil(&self) -> &str {
		match self {
			IdType::UserId => "@",
			IdType::RoomId => "!",
			IdType::EventId => "$",
			IdType::GroupId => "+",
			IdType::RoomAlias => "#",
			IdType::Unknown(x) => x.as_str(),
		}
	}

	/// Returns the type encoded as a type string
	pub fn to_type_str(&self) -> &str {
		match self {
			IdType::UserId => "u",
			IdType::RoomId => "roomid",
			IdType::EventId => "e",
			IdType::GroupId => "+", // Undefined so returning Sigil
			IdType::RoomAlias => "r",
			IdType::Unknown(x) => x.as_str(),
		}
	}
}

impl FromStr for IdType {
	type Err = ();

	/// Parses both Sigils and type strings into an MXID type
	fn from_str(s: &str) -> Result<Self, Self::Err> {
		Ok(match s {
			"@" | "u" | "user" => Self::UserId,
			"!" | "roomid" => Self::RoomId,
			"$" | "e" | "event" => Self::EventId,
			"+" => Self::GroupId,
			"#" | "r" | "room" => Self::RoomAlias,
			oth => Self::Unknown(oth.to_string()),
		})
	}
}

/// Represents an MXID, has a type and a body
#[derive(Debug)]
pub struct MatrixId {
	/// Type of Matrix ID
	pub id_type: IdType,
	/// Contents of Matrix ID
	pub body: String,
}

impl MatrixId {
	/// Creates a new Matrix ID
	pub fn new(id_type: IdType, body: String) -> Self {
		Self { id_type, body }
	}
}

impl Display for MatrixId {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		write!(f, "{}{}", self.id_type.to_sigil(), &self.body)
	}
}

#[cfg(feature = "ruma")]
impl TryInto<ruma_identifiers::UserId> for MatrixId {
	type Error = ();

	fn try_into(self) -> Result<ruma_identifiers::UserId, Self::Error> {
		if self.id_type == IdType::UserId {
			ruma_identifiers::UserId::try_from(self.to_string()).map_err(|_| ())
		} else {
			Err(())
		}
	}
}

#[cfg(feature = "ruma")]
impl TryInto<ruma_identifiers::RoomId> for MatrixId {
	type Error = ();

	fn try_into(self) -> Result<ruma_identifiers::RoomId, Self::Error> {
		if self.id_type == IdType::RoomId {
			ruma_identifiers::RoomId::try_from(self.to_string()).map_err(|_| ())
		} else {
			Err(())
		}
	}
}

#[cfg(feature = "ruma")]
impl TryInto<ruma_identifiers::RoomAliasId> for MatrixId {
	type Error = ();

	fn try_into(self) -> Result<ruma_identifiers::RoomAliasId, Self::Error> {
		if self.id_type == IdType::RoomAlias {
			ruma_identifiers::RoomAliasId::try_from(self.to_string()).map_err(|_| ())
		} else {
			Err(())
		}
	}
}

#[cfg(feature = "ruma")]
impl TryInto<ruma_identifiers::EventId> for MatrixId {
	type Error = ();

	fn try_into(self) -> Result<ruma_identifiers::EventId, Self::Error> {
		if self.id_type == IdType::EventId {
			ruma_identifiers::EventId::try_from(self.to_string()).map_err(|_| ())
		} else {
			Err(())
		}
	}
}

/// Actions specifiable by the `action` query parameter
#[derive(Copy, Clone, PartialEq, Eq, Debug)]
pub enum MatrixUriAction {
	/// Only valid for Matrix Users, client SHOULD open a [canonical direct chat](https://github.com/matrix-org/matrix-doc/pull/2199) with the user if supported otherwise they SHOULD open a direct chat window with the user
	Chat,
	/// Only valid for Matrix Rooms, client SHOULD attempt to join the room
	/// specified via standard CS API
	Join,
}

impl FromStr for MatrixUriAction {
	type Err = ();

	fn from_str(s: &str) -> Result<Self, Self::Err> {
		match s {
			"chat" => Ok(Self::Chat),
			"join" => Ok(Self::Join),
			_ => Err(()),
		}
	}
}

impl ToString for MatrixUriAction {
	fn to_string(&self) -> String {
		String::from(match self {
			MatrixUriAction::Chat => "chat",
			MatrixUriAction::Join => "join",
		})
	}
}

/// A Matrix URI
#[derive(Debug)]
pub struct MatrixUri {
	/// Primary MXID
	pub mxid: MatrixId,
	/// Secondary, optional, MXID
	pub child_mxid: Option<MatrixId>,
	query: HashMap<String, String>,
	/// Homeserver address order is preserved from URI
	pub routing: Vec<String>,

	/// "Reserved for future use" according to MSC
	pub authority: Option<String>,
	/// "Reserved for future use" according to MSC
	pub fragment: Option<String>,
}

/// Error that can happen when generating a [`MatrixUri`] from invalid
/// arguments.
#[derive(Debug)]
pub enum MatrixUriGenError {
	/// A child id was provided for an id type that doesn't accept one.
	UnexpectedChild,
}

impl Display for MatrixUriGenError {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		match self {
			MatrixUriGenError::UnexpectedChild => {
				write!(f, "Child matrix id provided for identifier type which doesn't accept one")
			}
		}
	}
}

impl std::error::Error for MatrixUriGenError {}

impl MatrixUri {
	/// Builds new Matrix URI
	pub fn new(
		mxid: MatrixId,
		child_mxid: Option<MatrixId>,
		query_params: Option<HashMap<String, String>>,
		routing: Option<Vec<String>>,
		action: Option<MatrixUriAction>,
	) -> Result<Self, MatrixUriGenError> {
		if child_mxid.is_some()
			&& (mxid.id_type != IdType::RoomAlias && mxid.id_type != IdType::RoomId)
		{
			return Err(MatrixUriGenError::UnexpectedChild);
		}

		let routing = {
			let mut out = Vec::new();

			if let Some(map) = query_params.as_ref() {
				if let Some(val) = map.get("via") {
					out.push(val.clone());
				}
			}

			if !out.is_empty() {
				if let Some(routing) = routing.as_ref() {
					for x in routing {
						out.push(x.clone());
					}
				}

				out
			} else {
				routing.unwrap_or_default()
			}
		};

		let query = {
			let mut query = match query_params {
				Some(query) => query,
				None => HashMap::new(),
			};

			if let Some(action) = action {
				query.insert(String::from("action"), action.to_string());
			}

			query
		};

		Ok(Self { mxid, child_mxid, query, routing, authority: None, fragment: None })
	}

	/// Returns the `action` key from query parameters
	pub fn action(&self) -> Option<MatrixUriAction> {
		if let Some(x) = self.query.get("action") {
			if let Ok(action) = MatrixUriAction::from_str(x.as_str()) {
				Some(action)
			} else {
				None
			}
		} else {
			None
		}
	}

	/// Constructs a query string from itself
	pub fn query_string(&self) -> String {
		let mut base = self.query.iter().fold(String::new(), |acc, (key, value)| {
			let encoded = (urlencoding::encode(key.as_str()), urlencoding::encode(value.as_str()));
			if acc.is_empty() {
				format!("{}={}", encoded.0, encoded.1)
			} else {
				format!("{}&{}={}", acc, encoded.0, encoded.1)
			}
		});

		for x in &self.routing {
			if !base.is_empty() {
				base += "&";
			}
			base = format!("{}via={}", base, urlencoding::encode(x.as_str()));
		}

		base
	}

	/// Builds a `matrix.to` URI
	pub fn matrix_to_string(&self) -> String {
		let mut base =
			format!("https://matrix.to/#/{}", urlencoding::encode(self.mxid.to_string().as_str()));

		if let Some(mxid) = self.child_mxid.as_ref() {
			base = format!("{}/{}", base, urlencoding::encode(mxid.to_string().as_str()));
		}

		let query_string = self.query_string();

		if !query_string.is_empty() {
			base = format!("{}?{}", base, query_string);
		}

		base
	}

	/// Builds a URI according to MSC2312 (`matrix://`)
	pub fn matrix_uri_string(&self) -> String {
		let mut base = String::from("matrix:");

		if let Some(auth) = self.authority.as_ref() {
			base = format!("{}//{}/", base, auth);
		}

		base = format!(
			"{}{}/{}",
			base,
			self.mxid.id_type.to_type_str(),
			urlencoding::encode(self.mxid.body.as_str())
		);

		if let Some(mxid) = self.child_mxid.as_ref() {
			base = format!(
				"{}/{}/{}",
				base,
				mxid.id_type.to_type_str(),
				urlencoding::encode(mxid.body.as_str())
			);
		}

		let query_string = self.query_string();

		if !query_string.is_empty() {
			base = format!("{}?{}", base, query_string);
		}

		if let Some(frag) = self.fragment.as_ref() {
			base = format!("{}#{}", base, frag);
		}

		base
	}
}

/// Errors encountered while parsing a URI
#[derive(Debug)]
pub enum MatrixUriParseError {
	/// Wrapper for [url::ParseError](https://docs.rs/url/2.2.1/url/enum.ParseError.html)
	ParseError(url::ParseError),
	/// Wrapper for [urlencoding::FromUrlEncodingError](https://docs.rs/urlencoding/1.1.1/urlencoding/enum.FromUrlEncodingError.html)
	DecodeError(urlencoding::FromUrlEncodingError),
	/// A URI path segment is empty
	EmptySegment,
	/// URI is missing path
	MissingPath,
	/// URI Schema is not recognized
	InvalidSchema,
	/// URI path has an invalid amount of segments
	InvalidSegmentCount,
	/// MXID Sigil is unrecognized
	InvalidSigil,
	/// URI is badly formatted
	InvalidUri,
}

fn parse_query(query: &str) -> (HashMap<String, String>, Vec<String>) {
	let mut routing: Vec<String> = Vec::new();
	let res = query
		.split('&')
		.filter_map(|x| {
			let idx = x.find('=')?;
			let sides = x.split_at(idx);

			let sides =
				(urlencoding::decode(sides.0).ok()?, urlencoding::decode(&sides.1[1..]).ok()?);

			if sides.0.as_str() == "via" {
				routing.push(sides.1);
				None
			} else {
				Some(sides)
			}
		})
		.collect();
	(res, routing)
}

impl FromStr for MatrixUri {
	type Err = MatrixUriParseError;

	/// Parses both `matrix.to` URLs and [MSC2312](https://github.com/matrix-org/matrix-doc/pull/2312) (`matrix://`) URIs
	fn from_str(s: &str) -> Result<Self, Self::Err> {
		let schema_send = match s.chars().position(|c| c == ':') {
			Some(offset) => offset,
			None => return Err(MatrixUriParseError::InvalidSchema),
		};

		let scheme = s[0..schema_send].to_ascii_lowercase();
		let scheme = scheme.as_str();

		if scheme == "http" || scheme == "https" {
			if !s[scheme.len()..].to_ascii_lowercase().starts_with("://matrix.to/#/") {
				return Err(MatrixUriParseError::InvalidUri);
			}

			let url = Url::parse(s.replacen("/#/", "/", 1).as_str())
				.map_err(MatrixUriParseError::ParseError)?;

			if url.host_str().map_or(true, |host| host.to_ascii_lowercase() != "matrix.to") {
				return Err(MatrixUriParseError::InvalidUri);
			}

			let path = &url.path()[1..];
			let path_split: Vec<&str> = path.split('/').collect();

			if path_split.len() != 1 && path_split.len() != 2 {
				return Err(MatrixUriParseError::InvalidSegmentCount);
			}

			let seg_1_decoded =
				urlencoding::decode(path_split[0]).map_err(MatrixUriParseError::DecodeError)?;

			let sigil_1 = match IdType::from_str(&seg_1_decoded[0..1]) {
				Ok(sigil) => sigil,
				Err(_) => return Err(MatrixUriParseError::InvalidSigil),
			};

			let body_1 = &seg_1_decoded[1..];

			if body_1.is_empty() {
				return Err(MatrixUriParseError::EmptySegment);
			}

			let mxid = MatrixId::new(sigil_1, body_1.to_string());

			let child_mxid = {
				if path_split.len() == 2
					&& (mxid.id_type == IdType::RoomAlias || mxid.id_type == IdType::RoomId)
				{
					let seg_2_decoded = urlencoding::decode(path_split[1])
						.map_err(MatrixUriParseError::DecodeError)?;

					let sigil_2 = match IdType::from_str(&seg_2_decoded[0..1]) {
						Ok(sigil) => sigil,
						Err(_) => return Err(MatrixUriParseError::InvalidSigil),
					};

					if sigil_2 != IdType::EventId {
						return Err(MatrixUriParseError::InvalidSigil);
					}

					let body_2 = &seg_2_decoded[1..];

					if body_2.is_empty() {
						return Err(MatrixUriParseError::EmptySegment);
					}

					Some(MatrixId::new(sigil_2, body_2.to_string()))
				} else {
					None
				}
			};

			let query = url.query().unwrap_or("").to_string();

			let (query, routing) = parse_query(query.as_str());

			Ok(MatrixUri { mxid, child_mxid, query, routing, authority: None, fragment: None })
		} else if scheme == "matrix" {
			lazy_static! {
				static ref M_URI_REGEX: Regex = Regex::new(
					r":(?://(?P<authority>[^/]*)/)?(?P<path>[^?]+)(?:\?(?P<query>[^#]*))?(?:#(?P<fragment>.*))?"
				)
				.unwrap_or_else(|e| panic!("Invalid regex: {}", e));
			}

			let captures: Captures = match M_URI_REGEX.captures(s) {
				Some(x) => x,
				None => return Err(MatrixUriParseError::InvalidUri),
			};

			let path = match captures.name("path") {
				Some(x) => x.as_str().to_string(),
				None => return Err(MatrixUriParseError::MissingPath),
			};

			let authority = captures.name("authority").map(|x| x.as_str().to_string());
			let fragment = captures.name("fragment").map(|x| x.as_str().to_string());
			let query =
				captures.name("query").map(|x| x.as_str().to_string()).unwrap_or_else(String::new);

			let path_split: Vec<&str> = path.split('/').collect();

			if path_split.len() != 2 && path_split.len() != 4 {
				return Err(MatrixUriParseError::InvalidSegmentCount);
			}

			let sigil_1 = match IdType::from_str(
				urlencoding::decode(path_split[0])
					.map_err(MatrixUriParseError::DecodeError)?
					.as_str(),
			) {
				Ok(sigil) => sigil,
				Err(_) => return Err(MatrixUriParseError::InvalidSigil),
			};

			if ![IdType::UserId, IdType::RoomAlias, IdType::RoomId].contains(&sigil_1) {
				return Err(MatrixUriParseError::InvalidSigil);
			}

			let body_1 =
				urlencoding::decode(path_split[1]).map_err(MatrixUriParseError::DecodeError)?;

			if body_1.is_empty() {
				return Err(MatrixUriParseError::EmptySegment);
			}

			let mxid = MatrixId::new(sigil_1, body_1);

			let child_mxid = {
				if path_split.len() == 4
					&& (mxid.id_type == IdType::RoomAlias || mxid.id_type == IdType::RoomId)
				{
					let sigil_2 = match IdType::from_str(
						urlencoding::decode(path_split[2])
							.map_err(MatrixUriParseError::DecodeError)?
							.as_str(),
					) {
						Ok(sigil) => sigil,
						Err(_) => return Err(MatrixUriParseError::InvalidSigil),
					};

					if sigil_2 != IdType::EventId {
						return Err(MatrixUriParseError::InvalidSigil);
					}

					let body_2 = urlencoding::decode(path_split[3])
						.map_err(MatrixUriParseError::DecodeError)?;

					if body_2.is_empty() {
						return Err(MatrixUriParseError::EmptySegment);
					}

					Some(MatrixId::new(sigil_2, body_2))
				} else {
					None
				}
			};

			let (query, routing) = parse_query(query.as_str());

			Ok(MatrixUri { mxid, child_mxid, query, routing, authority, fragment })
		} else {
			Err(MatrixUriParseError::InvalidSchema)
		}
	}
}

#[allow(clippy::unwrap_used)]
#[cfg(test)]
mod tests {
	use std::str::FromStr;

	#[cfg(feature = "ruma")]
	#[test]
	fn ruma() {
		let uri = crate::MatrixUri::new(
			crate::MatrixId::new(crate::IdType::UserId, String::from("cute:some.url")),
			None,
			None,
			None,
		)
		.unwrap();

		let user_id: ruma_identifiers::UserId = uri.mxid.try_into().unwrap();

		assert_eq!(user_id.as_str(), "@cute:some.url");

		let uri = crate::MatrixUri::new(
			crate::MatrixId::new(crate::IdType::RoomId, String::from("rid:example.org")),
			None,
			None,
			None,
		)
		.unwrap();

		let room_id: ruma_identifiers::RoomId = uri.mxid.try_into().unwrap();

		assert_eq!(room_id.as_str(), "!rid:example.org");

		let uri = crate::MatrixUri::new(
			crate::MatrixId::new(crate::IdType::RoomAlias, String::from("us:example.org")),
			Some(crate::MatrixId::new(crate::IdType::EventId, String::from("lol823y4bcp3qo4"))),
			None,
			None,
		)
		.unwrap();

		let room_alias_id: ruma_identifiers::RoomAliasId = uri.mxid.try_into().unwrap();
		let event_id: ruma_identifiers::EventId = uri.child_mxid.unwrap().try_into().unwrap();

		assert_eq!(room_alias_id.as_str(), "#us:example.org");
		assert_eq!(event_id.as_str(), "$lol823y4bcp3qo4")
	}

	#[test]
	fn building() {
		let uri = crate::MatrixUri::new(
			crate::MatrixId::new(crate::IdType::UserId, String::from("cute:some.url")),
			None,
			None,
			Some(vec![String::from("headpat.services"), String::from("cute.local")]),
			Some(crate::MatrixUriAction::Chat),
		)
		.unwrap();

		assert_eq!(uri.mxid.to_string(), String::from("@cute:some.url"));
		assert_eq!(
			uri.query_string(),
			String::from("action=chat&via=headpat.services&via=cute.local")
		);
		assert_eq!(uri.matrix_to_string(), String::from("https://matrix.to/#/%40cute%3Asome.url?action=chat&via=headpat.services&via=cute.local"));
		assert_eq!(
			uri.matrix_uri_string(),
			String::from(
				"matrix:u/cute%3Asome.url?action=chat&via=headpat.services&via=cute.local"
			)
		);

		let res = crate::MatrixUri::from_str(uri.matrix_to_string().as_str());

		assert!(res.is_ok());

		let uri2 = res.unwrap();

		assert_eq!(uri.matrix_uri_string(), uri2.matrix_uri_string());
		assert_eq!(uri.matrix_to_string(), uri2.matrix_to_string());
	}

	#[test]
	fn matrix_url_parsing() {
		let uri = crate::MatrixUri::from_str("matrix:r/someroom:example.org/e/Arbitrary_Event_Id")
			.unwrap();

		assert_eq!(uri.mxid.id_type, crate::IdType::RoomAlias);
		assert_eq!(uri.mxid.body, String::from("someroom:example.org"));
		assert_eq!(uri.mxid.to_string(), String::from("#someroom:example.org"));

		assert!(uri.child_mxid.is_some());
		assert_eq!(uri.child_mxid.as_ref().unwrap().id_type, crate::IdType::EventId);
		assert_eq!(uri.child_mxid.as_ref().unwrap().body, String::from("Arbitrary_Event_Id"));
		assert_eq!(
			uri.child_mxid.as_ref().unwrap().to_string(),
			String::from("$Arbitrary_Event_Id")
		);

		let uri = crate::MatrixUri::from_str("matrix:u/her:example.org?action=chat").unwrap();

		assert_eq!(uri.mxid.id_type, crate::IdType::UserId);
		assert_eq!(uri.mxid.body, String::from("her:example.org"));
		assert_eq!(uri.mxid.to_string(), String::from("@her:example.org"));

		let action_res = uri.action();

		assert!(action_res.is_some());
		assert_eq!(action_res.unwrap(), crate::MatrixUriAction::Chat);

		let uri = crate::MatrixUri::from_str(
			"matrix:roomid/rid:example.org?action=join&via=example2.org",
		)
		.unwrap();

		assert_eq!(uri.mxid.id_type, crate::IdType::RoomId);
		assert_eq!(uri.mxid.body, String::from("rid:example.org"));
		assert_eq!(uri.mxid.to_string(), String::from("!rid:example.org"));

		assert_eq!(uri.routing.len(), 1);
		assert_eq!(uri.routing[0], String::from("example2.org"));

		let action_res = uri.action();

		assert!(action_res.is_some());
		assert_eq!(action_res.unwrap(), crate::MatrixUriAction::Join);
	}

	#[test]
	fn https_url_parsing() {
		let url = crate::MatrixUri::from_str(
			"https://matrix.to/#/%23somewhere:example.org/%24event%3Aexample.org?test123",
		)
		.unwrap();

		assert_eq!(url.mxid.id_type, crate::IdType::RoomAlias);
		assert_eq!(url.mxid.body, String::from("somewhere:example.org"));
		assert_eq!(url.mxid.to_string(), String::from("#somewhere:example.org"));

		assert!(url.child_mxid.is_some());
		assert_eq!(url.child_mxid.as_ref().unwrap().id_type, crate::IdType::EventId);
		assert_eq!(url.child_mxid.as_ref().unwrap().body, String::from("event:example.org"));
		assert_eq!(
			url.child_mxid.as_ref().unwrap().to_string(),
			String::from("$event:example.org")
		);

		let url =
			crate::MatrixUri::from_str("https://matrix.to/#/%40alice%3Aexample.org?query=some")
				.unwrap();

		assert_eq!(url.mxid.id_type, crate::IdType::UserId);
		assert_eq!(url.mxid.body, String::from("alice:example.org"));

		let query_opt = url.query.get("query");
		assert!(query_opt.is_some());
		assert_eq!(query_opt.unwrap(), "some");
	}
}
