use std::fs::File;
use std::io::{Read, self, Write};
use std::path::PathBuf;

use anyhow::{Context, Result};
use comrak::{
    nodes::{AstNode, NodeValue},
    parse_document, Arena, ComrakOptions,
};
use structopt::StructOpt;

#[derive(Debug, StructOpt)]
struct Opt {
    /// The language to filter for
    #[structopt(long = "lang", short = "l")]
    lang: Option<String>,

    /// The file to extract from
    file: PathBuf,
}

fn main() -> Result<()> {
    let opt = Opt::from_args();

    let contents = {
        let mut file = File::open(&opt.file)
            .with_context(|| format!("could not open target file {:?}", opt.file))?;
        let mut contents = String::new();
        file.read_to_string(&mut contents)
            .with_context(|| format!("could not read target file {:?}", opt.file))?;
        contents
    };

    let mut stdout = io::stdout();

    let arena = Arena::new();
    let ast_node = parse_document(&arena, &contents, &ComrakOptions::default());
    visit_ast(&opt, &mut stdout, ast_node)?;

    Ok(())
}

fn visit_ast<'a, W: Write>(opt: &Opt, out: &mut W, node: &'a AstNode<'a>) -> Result<()> {
    let ast = node.data.borrow();
    match &ast.value {
        NodeValue::CodeBlock(code_block) => {
            let should_emit = match &opt.lang {
                Some(lang) => lang.as_bytes() == &code_block.info,
                None => true,
            };

            if should_emit {
                out.write(&code_block.literal)?;
            }
        }
        NodeValue::List(_) | NodeValue::Item(_) | NodeValue::Document => {
            // recurse
            for child in node.children() {
                visit_ast(opt, out, child)?;
            }
        }
        NodeValue::Heading(_) | NodeValue::BlockQuote | NodeValue::Paragraph => {
            // don't do anything
        }
        _ => todo!("unsupported type {:?}", ast.value),
    }

    Ok(())
}
