extern crate regex;

use std::ops::Add;

const STX:char = '\u{0002}';  // Use STX ("Start of text") for start-of-placeholder
const ETX:char = '\u{0003}'; // Use ETX ("End of text") for end-of-placeholder
const INLINE_PLACEHOLDER_PREFIX: &str = "\u{0002}klzzwxh:";
//const INLINE_PLACEHOLDER:String = String::from(INLINE_PLACEHOLDER_PREFIX).add("%s") + &*ETX.to_string();
const INLINE_PLACEHOLDER_RE: &str = "\u{0002}klzzwxh:([0-9]+)\u{0003}";
const AMP_SUBSTITUTE: &str = "\u{0002}amp\u{0003}";
const HTML_PLACEHOLDER_START: &str = "\u{0002}wzxhzdk:";
const HTML_PLACEHOLDER_END: &str = "\u{0003}";
const HTML_PLACEHOLDER_RE: &str = "\u{0003}wzxhzdk:([0-9]+)\u{0003}";
const TAG_PLACEHOLDER_START: &str = "\u{0002}hzzhzkh:";
const TAG_PLACEHOLDER_END: &str = "\u{0003}";

pub trait Processor {
    fn new(md:String);
}

pub struct Tag {
    tag: String,
    attrs: Vec<String>,
    left_index: u32,
    right_index: u32,
}
pub struct HtmlStash {
    html_counter:u32, // for counting inline html segments
    raw_html_blocks:Vec<String>,
    tag_counter:u32,
    tag_data:Vec<Tag>,
}

impl HtmlStash {
    pub fn new() -> HtmlStash{
        return HtmlStash {
            html_counter:0,
            raw_html_blocks:vec!(),
            tag_counter:0,
            tag_data:vec!(),
        }
    }

    /// Saves an HTML segment for later reinsertion. Returns a
    /// placeholder string that needs to be inserted into the
    /// document.
    /// ## Keyword arguments:
    /// - html: an html segment
    /// ## Returns :
    /// - a placeholder string
    pub fn store(&mut self, html:String) -> String {
        self.raw_html_blocks.push(html);
        let placeholder = self.get_placeholder(&*format!("{}", self.html_counter));
        self.html_counter += 1;
        return placeholder;
    }

    pub fn reset(&mut self) {
        self.html_counter = 0;
        self.raw_html_blocks = vec!();
    }
    pub fn get_placeholder(&self, key:&str) -> String {
        return String::from(HTML_PLACEHOLDER_START)
            .add(key).add(HTML_PLACEHOLDER_END);
    }
    /// Store tag data and return a placeholder.
    pub fn store_tag(&mut self, tag:String, attrs:Vec<String>, left_index:u32, right_index:u32) -> String {
        self.tag_data.push(Tag {
            tag,
            attrs,
            left_index,
            right_index,
        });
        let placeholder:String = String::from(TAG_PLACEHOLDER_START)
            .add(&*format!("{}", self.tag_counter))
            .add(TAG_PLACEHOLDER_END);
        self.tag_counter += 1; //equal to the tag's index in self.tag_data
        return placeholder
    }
}