//! Parses and manipulates problem yaml

use crate::result::MapmErr;
use crate::result::MapmErr::*;
use crate::result::MapmResult;

use crate::template::Template;
use Filter::*;
use FilterAction::*;
use Views::*;

use std::fs;
use std::path::Path;

use std::collections::HashMap;

use serde::{Deserialize, Serialize};

use serde_yaml::Value;

pub type Vars = HashMap<String, String>;
pub type Solutions = Vec<HashMap<String, String>>;

#[derive(Debug, Serialize, Deserialize)]
struct SerializedProblem {
    pub solutions: Option<Solutions>,
    #[serde(flatten)]
    pub vars: HashMap<String, Value>,
}

#[derive(Debug, PartialEq, Clone)]
pub struct Problem {
    pub vars: Vars,
    pub solutions: Solutions,
}

/// Defines types of filters for problems
///
/// Only the "Exists" variant may be used for the special key "solutions".
///
/// For the Gt, Lt, Ge, Le conditions, whoever is working with the problem files (either through CLI utils or a GUI) is responsible for ensuring that any key being compared with Gt, Lt, Ge, Le is a non-negative integer

#[derive(Clone)]
pub enum Filter {
    Exists { key: String },
    Eq { key: String, val: String },
    Gt { key: String, val: u32 },
    Lt { key: String, val: u32 },
    Ge { key: String, val: u32 },
    Le { key: String, val: u32 },
}

/// Defines the actions for filtering problems
///
/// Positive means "keep if this condition is satisfied", negative means "keep if this condition is **not** satisfied"
#[derive(Clone)]
pub enum FilterAction {
    Positive(Filter),
    Negative(Filter),
}

/// Views *only* shown strings or views everything *but* hidden strings
#[derive(Clone)]
pub enum Views {
    Show(Vec<String>),
    Hide(Vec<String>),
}

/// Gets a problem from passed in filepath and name of problem

pub fn fetch_problem(problem_name: &str, problem_dir: &Path) -> MapmResult<Problem> {
    let problem_src = &problem_dir.join(&[problem_name, ".yml"].concat());
    match fs::read_to_string(problem_src) {
        Ok(problem_yaml) => parse_problem_yaml(&problem_yaml),
        Err(_) => Err(ProblemErr(format!(
            "Could not read problem `{}` from {:?}",
            problem_name, problem_src
        ))),
    }
}

impl Problem {
    /// Checks if a problem passes or fails a certain filter
    ///
    /// For Gt, it checks if the value of the problem is greater than the value passed in the filter (not the other way around). The same is true of Lt, Ge, Le.
    ///
    /// # Usage
    ///
    /// ```
    /// use mapm::problem::Problem;
    /// use mapm::problem::Filter;
    /// use std::collections::HashMap;
    ///
    /// let mut vars = HashMap::new();
    /// vars.insert(String::from("problem"), String::from("What is $1+1$?"));
    /// vars.insert(String::from("author"), String::from("Dennis Chen"));
    /// vars.insert(String::from("difficulty"), String::from("5"));
    ///
    /// let mut solution_one = HashMap::new();
    /// solution_one.insert(String::from("text"), String::from("It's probably $2$."));
    /// solution_one.insert(String::from("author"), String::from("Dennis Chen"));
    /// let mut solution_two = HashMap::new();
    /// solution_two.insert(String::from("text"), String::from("The answer is $2$, but my proof is too small to fit into the margin."));
    /// solution_two.insert(String::from("author"), String::from("Pierre de Fermat"));
    ///
    /// let mut solutions = Vec::from([solution_one, solution_two]);
    ///
    /// let problem = Problem { vars, solutions };
    ///
    /// assert_eq!(problem.try_filter(Filter::Exists{key: String::from("subject")}), false);
    /// assert_eq!(problem.try_filter(Filter::Exists{key: String::from("solutions")}), true);
    ///
    /// assert_eq!(problem.try_filter(Filter::Gt{key: String::from("difficulty"), val: 5}), false);
    /// assert_eq!(problem.try_filter(Filter::Ge{key: String::from("difficulty"), val: 5}), true);
    /// assert_eq!(problem.try_filter(Filter::Lt{key: String::from("difficulty"), val: 5}), false);
    /// assert_eq!(problem.try_filter(Filter::Le{key: String::from("difficulty"), val: 5}), true);
    ///
    /// ```

    pub fn try_filter(&self, filter: Filter) -> bool {
        match filter {
            Exists { key } => {
                if key == "solutions" {
                    self.solutions.len() > 0
                } else {
                    self.vars.contains_key(&key)
                }
            }
            Eq { key, val } => match self.vars.get(&key) {
                Some(problem_value) => problem_value == &val,
                None => false,
            },
            Gt { key, val } => match self.vars.get(&key) {
                Some(problem_value) => match problem_value.parse::<u32>() {
                    Ok(problem_value_u32) => problem_value_u32 > val,
                    Err(_) => false,
                },
                None => false,
            },
            Lt { key, val } => match self.vars.get(&key) {
                Some(problem_value) => match problem_value.parse::<u32>() {
                    Ok(problem_value_u32) => problem_value_u32 < val,
                    Err(_) => false,
                },
                None => false,
            },
            Ge { key, val } => match self.vars.get(&key) {
                Some(problem_value) => match problem_value.parse::<u32>() {
                    Ok(problem_value_u32) => problem_value_u32 >= val,
                    Err(_) => false,
                },
                None => false,
            },
            Le { key, val } => match self.vars.get(&key) {
                Some(problem_value) => match problem_value.parse::<u32>() {
                    Ok(problem_value_u32) => problem_value_u32 <= val,
                    Err(_) => false,
                },
                None => false,
            },
        }
    }

    /// Checks whether a problem satisfies a set of filters, and if it does, return it; otherwise return `None`
    ///
    /// If `filters` is empty, then every problem will pass the filter.

    pub fn filter(self, filters: Vec<FilterAction>) -> Option<Problem> {
        for filter_action in filters {
            match filter_action {
                Positive(filter) => {
                    if !self.try_filter(filter) {
                        return None;
                    }
                }
                Negative(filter) => {
                    if self.try_filter(filter) {
                        return None;
                    }
                }
            }
        }
        Some(self)
    }

    /// Show only the keys passed in or everything but the keys passed in to a problem
    ///
    /// # Usage
    ///
    /// ```
    /// use mapm::problem::Views;
    /// use mapm::problem::Views::{Show, Hide};
    /// use mapm::problem::Problem;
    /// use mapm::problem::parse_problem_yaml;
    /// use mapm::problem::Filter;
    /// use std::collections::HashMap;
    ///
    /// let problem_yaml = "problem: What is $1+1$?
    /// author: Dennis Chen
    /// answer: 2
    /// solutions:
    ///   - text: It's $2$.
    ///     author: Alexander
    /// ";
    ///
    /// let problem = parse_problem_yaml(problem_yaml).unwrap();
    ///
    /// let show: Views = Show(Vec::from([String::from("author"), String::from("answer"), String::from("solutions")]));
    /// let show_filtered_problem = problem.clone().filter_keys(show);
    /// assert_eq!(show_filtered_problem.0, HashMap::from([(String::from("author"), String::from("Dennis Chen")), (String::from("answer"), String::from("2"))]));
    /// assert_eq!(show_filtered_problem.1, Some(Vec::from([
    ///     HashMap::from([(String::from("text"), String::from("It's $2$.")), (String::from("author"), String::from("Alexander"))])
    /// ])));
    ///
    /// let hide: Views = Hide(Vec::from([String::from("solutions"), String::from("author")]));
    /// let hide_filtered_problem = problem.clone().filter_keys(hide);
    /// assert_eq!(hide_filtered_problem.0, HashMap::from([(String::from("problem"), String::from("What is $1+1$?")), (String::from("answer"), String::from("2"))]));
    /// assert_eq!(hide_filtered_problem.1, None);
    /// ```

    pub fn filter_keys(self, views: Views) -> (Vars, Option<Solutions>) {
        match views {
            Show(tags) => {
                let mut vars: Vars = HashMap::new();
                for (key, val) in self.vars {
                    if key != "solutions" && tags.contains(&key) {
                        vars.insert(key, val);
                    }
                }
                if tags.contains(&"solutions".to_string()) {
                    (vars, Some(self.solutions))
                } else {
                    (vars, None)
                }
            }
            Hide(tags) => {
                let mut vars: Vars = HashMap::new();
                for (key, val) in self.vars {
                    if key != "solutions" && !tags.contains(&key) {
                        vars.insert(key, val);
                    }
                }
                if !tags.contains(&"solutions".to_string()) {
                    (vars, Some(self.solutions))
                } else {
                    (vars, None)
                }
            }
        }
    }

    /// Checks if a problem contains all the variables in a template
    ///
    /// Returns None if the problem successfully passes the test, returns Some(MapmErr) otherwise
    ///
    /// # Usage
    ///
    /// ## Expected success
    ///
    /// ```
    /// use mapm::problem::*;
    /// use mapm::template::*;
    /// let problem_yaml = "problem: What is $1+1$?
    /// solutions:
    ///   - text: Some say the answer is $2$.
    ///     author: Dennis Chen
    /// ";
    ///
    /// let template_yaml = "engine: pdflatex
    /// texfiles:
    ///   problems.tex: ${title}.PDF
    ///   solutions.tex: ${title}-sols.PDF
    /// problem_count: 1
    /// vars:
    ///   - title
    ///   - year
    /// problemvars:
    ///   - problem
    /// solutionvars:
    ///   - text
    ///   - author
    /// ";
    ///
    /// let problem = parse_problem_yaml(problem_yaml).unwrap();
    /// let template = parse_template_yaml(template_yaml).unwrap();
    ///
    /// assert!(problem.check_template(&template).is_none());
    /// ```
    ///
    /// ## Expected failure
    ///
    /// ```
    /// use mapm::problem::*;
    /// use mapm::template::*;
    /// use mapm::result::MapmErr;
    /// use mapm::result::MapmErr::*;
    /// let problem_yaml = "problem: What is $1+1$?
    /// solutions:
    ///   - text: Some say the answer is $2$.
    ///     author: Dennis Chen
    /// ";
    ///
    /// let template_yaml = "engine: pdflatex
    /// texfiles:
    ///   problems.tex: ${title}.PDF
    ///   solutions.tex: ${title}-sols.PDF
    /// vars:
    ///   - title
    ///   - year
    /// problem_count: 1
    /// problemvars:
    ///   - problem
    ///   - author
    /// solutionvars:
    ///   - text
    ///   - author
    /// ";
    ///
    /// let problem = parse_problem_yaml(problem_yaml).unwrap();
    /// let template = parse_template_yaml(template_yaml).unwrap();
    ///
    /// let template_check = problem.check_template(&template).unwrap();
    ///
    /// assert_eq!(template_check.len(), 1);
    /// match &template_check[0] {
    ///     ProblemErr(err) => {
    ///         assert_eq!(err, "Does not contain key `author`");
    ///     }
    ///     _ => {
    ///         panic!("MapmErr type is not ProblemErr");
    ///     }
    /// }
    /// ```

    pub fn check_template(&self, template: &Template) -> Option<Vec<MapmErr>> {
        let mut mapm_errs: Vec<MapmErr> = Vec::new();

        for problemvar in &template.problemvars {
            if !self.vars.contains_key(problemvar) {
                mapm_errs.push(ProblemErr(
                    ["Does not contain key `", &problemvar, "`"].concat(),
                ));
            }
        }
        let mut index: u32 = 0;
        for map in &self.solutions {
            index += 1;
            for solutionvar in &template.solutionvars {
                if !map.contains_key(solutionvar) {
                    mapm_errs.push(SolutionErr(
                        [
                            "Solution `",
                            &index.to_string(),
                            "` does not contain key `",
                            &solutionvar,
                            "`",
                        ]
                        .concat(),
                    ));
                }
            }
        }
        if mapm_errs.len() > 0 {
            return Some(mapm_errs);
        } else {
            return None;
        }
    }
}

///
/// # Usage
///
/// ```
/// use mapm::problem::Problem;
/// use mapm::problem::parse_problem_yaml;
/// use std::collections::HashMap;
/// let yaml = "problem: What is $1+1$?
/// author: Dennis Chen
/// solutions:
///   - text: It's probably $2$.
///     author: Dennis Chen
///   - text: The answer is $2$, but my proof is too small to fit into the margin.
///     author: Pierre de Fermat
/// ";
/// let problem = parse_problem_yaml(yaml).unwrap();
///
/// let mut vars = HashMap::new();
/// vars.insert(String::from("problem"), String::from("What is $1+1$?"));
/// vars.insert(String::from("author"), String::from("Dennis Chen"));
///
/// let mut solution_one = HashMap::new();
/// solution_one.insert(String::from("text"), String::from("It's probably $2$."));
/// solution_one.insert(String::from("author"), String::from("Dennis Chen"));
/// let mut solution_two = HashMap::new();
/// solution_two.insert(String::from("text"), String::from("The answer is $2$, but my proof is too small to fit into the margin."));
/// solution_two.insert(String::from("author"), String::from("Pierre de Fermat"));
///
/// let mut solutions = Vec::from([solution_one, solution_two]);
///
/// assert_eq!(problem.vars, vars);
/// assert_eq!(problem.solutions, solutions);
/// ```

pub fn parse_problem_yaml(yaml: &str) -> MapmResult<Problem> {
    match serde_yaml::from_str::<SerializedProblem>(yaml) {
        Ok(problem) => {
            let mut vars: Vars = HashMap::new();
            for (key, val) in problem.vars {
                match val {
                    Value::String(val) => {
                        vars.insert(key, val);
                    }
                    Value::Number(val) => {
                        vars.insert(key, val.to_string());
                    }
                    Value::Bool(val) => {
                        match val {
                            true => vars.insert(key, String::from("true")),
                            false => vars.insert(key, String::from("false")),
                        };
                    }
                    _ => {
                        return Err(ProblemErr(["Could not parse key `", &key, "`"].concat()));
                    }
                }
            }

            let solutions: Solutions = match problem.solutions {
                Some(solutions) => solutions,
                None => Vec::new(),
            };

            return Ok(Problem { vars, solutions });
        }
        Err(err) => {
            return Err(ProblemErr(err.to_string()));
        }
    }
}

/// Converts a vector of problems into TeX files and writes them in the current working directory
///
/// Internal function used for compiling vectors of problems and for compiling contests
///
/// Returns a header TeX string as well (for contest compilation if necessary)

pub(crate) fn write_as_tex(problem_vec: Vec<Problem>) -> String {
    let mut problem_number = 0;
    let mut headers = String::new();

    for problem in problem_vec {
        problem_number += 1;
        headers.push_str(
            &[
                "\\expandafter\\def\\csname mapm@solcount@",
                &problem_number.to_string(),
                "\\endcsname{",
                &problem.solutions.len().to_string(),
                "}",
            ]
            .concat(),
        );
        for (key, val) in &problem.vars {
            let filename = &["mapm-prob-", &problem_number.to_string(), "-", key, ".tex"].concat();
            fs::write(filename, val).expect(&["Could not write to `", &filename, "`"].concat());
        }
        for (pos, map) in problem.solutions.iter().enumerate() {
            for (key, val) in map {
                let solution_number = &pos + 1;
                let filename = &[
                    "mapm-sol-",
                    &problem_number.to_string(),
                    "-",
                    &solution_number.to_string(),
                    "-",
                    key,
                    ".tex",
                ]
                .concat();
                fs::write(filename, val).expect(&["Could not write to `", &filename, "`"].concat());
            }
        }
    }
    headers
}
