use std::fs;
use std::path::{Component, Path, PathBuf};

// Removes the prefix of a path.
//
// If it begins with a root, it removes the root AND the desecendant of the root.

fn remove_prefix<P>(from: P) -> PathBuf
where
    P: AsRef<Path>,
{
    let components = from.as_ref().components();
    let mut add: bool = false; // When this is true, we have finisehd traversing the prefix and can continue adding stuff to our new components
    let mut new_components = Vec::new();
    for component in components {
        if add {
            new_components.push(component);
        }
        if component != Component::RootDir {
            add = true;
        }
    }
    new_components.iter().collect::<PathBuf>()
}
#[test]
fn test_remove_prefix() {
    assert_eq!(remove_prefix("foo/bar"), Path::new("bar"));
    assert_eq!(remove_prefix("foo/bar/baz"), Path::new("bar/baz"));
    assert_eq!(remove_prefix("/foo/bar"), Path::new("bar"));
    assert_eq!(remove_prefix("/foo/bar/baz"), Path::new("bar/baz"));
}

/// Copies a directory into a destination directory,
/// ignoring any "dotfiles" which begin with a `.`
///
/// If the `to` directory does not exist, it will be created.
/// If it does exist, files inside will be overwritten,
/// but any remaining residual files will not be deleted.
///
/// If `to` exists but is not a directory (for instance, if it is a file),
/// this function will return an error.
///
/// You should always have `originator` be `true` when calling the function yourself,
/// `false` is for recursive calls.

pub(crate) fn copy_dir_ignore_dots<P, Q>(from: P, to: Q) -> std::io::Result<()>
where
    P: AsRef<Path>,
    Q: AsRef<Path>,
{
    let from = from.as_ref();
    let to = to.as_ref();
    if !to.is_dir() {
        if to.exists() {
            let err = &format!(
                "Destination path {:?} exists but is a directory; please move it to fix this issue.",
                to,
            );
            return Err(std::io::Error::new(std::io::ErrorKind::Other, err.as_str()));
        }
        match std::fs::create_dir(&to) {
            Err(e) => return Err(e),
            Ok(_) => {}
        }
    }
    for entry in fs::read_dir(from)? {
        let entry = entry?;
        let path = &entry.path();
        let destination = &to.join(remove_prefix(path));
        if path.is_file() || path.is_symlink() {
            fs::copy(path, destination)
                .expect(&format!("Failed to copy {:?} to {:?}", path, destination,));
        } else {
            if !destination.is_dir() {
                if destination.exists() {
                    let err = &format!(
                        "Destination path {:?} exists but is a directory; please move it to fix this issue.",
                        destination,
                    );
                    return Err(std::io::Error::new(std::io::ErrorKind::Other, err.as_str()));
                }
                match std::fs::create_dir(destination) {
                    Err(e) => return Err(e),
                    Ok(_) => {}
                }
            }
            copy_dir_ignore_dots(path, to)?
        }
    }
    Ok(())
}
