//! Parses and manipulates contest yaml

use std::collections::HashMap;

use crate::problem::*;

use crate::result::MapmErr;
use crate::result::MapmErr::*;
use crate::result::MapmResult::*;

use crate::template::fetch_template_config;
use crate::template::Template;

use crate::utils::copy::copy_dir_ignore_dots;

use std::env;
use std::fs;
use std::path::Path;
use std::process::Command;

use serde::{Deserialize, Serialize};

pub struct Contest {
    pub template: (String, Template),
    pub problems: Vec<(String, Problem)>,
    pub vars: HashMap<String, String>,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
struct UnresolvedContest {
    template: String,
    problems: Vec<String>,
    vars: HashMap<String, String>,
}

pub struct ContestResult {
    pub contest: Option<Contest>,
    pub contest_err: Option<MapmErr>,
    pub problem_errs: Option<Vec<(String, Vec<MapmErr>)>>,
}

/// Parses contest yaml into full-fledged contest, reading problem yamls based on `problem_dir` from the corresponding problem names in the contest file
///
/// MapmResult<Contest> will return with a ContestErr if the contest yaml cannot be parsed or does not satisfy the template conditions
///
/// Option<Vec<MapmErr>> will return with Some if any of the problems do not satisfy the template conditions, with the problem name being matched to the errors

pub fn parse_contest_yaml(yaml: &str, problem_dir: &Path, template_dir: &Path) -> ContestResult {
    match serde_yaml::from_str::<UnresolvedContest>(yaml) {
        Ok(unresolved_contest) => {
            match fetch_template_config(&unresolved_contest.template, template_dir) {
                Success(template) => {
                    if template.problem_count
                        != u32::try_from(unresolved_contest.problems.len()).unwrap()
                    {
                        return ContestResult {
                            contest: None,
                            contest_err: Some(TemplateErr(String::from("The number of problems in the contest yaml is not equivalent to `problem_count` in the template yaml"))),
                            problem_errs: None
                        };
                    }

                    let mut problems: Vec<(String, Problem)> = Vec::new();
                    let mut problem_errs: Vec<(String, Vec<MapmErr>)> = Vec::new();

                    for problem_name in unresolved_contest.problems {
                        let problem = fetch_problem(&problem_name, problem_dir);
                        match problem {
                            Success(problem) => {
                                problems.push((problem_name, problem));
                            }
                            Fail(err) => {
                                let mut errs = Vec::new();
                                errs.push(err);
                                problem_errs.push((problem_name, errs));
                            }
                        }
                    }
                    for problem in &problems {
                        match problem.1.check_template(&template) {
                            Some(errs) => {
                                problem_errs.push((problem.0.clone(), errs));
                            }
                            None => {}
                        }
                    }

                    let contest = Contest {
                        template: (unresolved_contest.template, template),
                        problems,
                        vars: unresolved_contest.vars,
                    };

                    let problem_errs_opt: Option<Vec<(String, Vec<MapmErr>)>>;

                    if problem_errs.len() == 0 {
                        problem_errs_opt = None;
                    } else {
                        problem_errs_opt = Some(problem_errs);
                    }

                    let contest_opt: Option<Contest>;

                    if matches!(contest.check_template(), None) && matches!(problem_errs_opt, None)
                    {
                        contest_opt = Some(contest);
                    } else {
                        contest_opt = None;
                    }

                    ContestResult {
                        contest: contest_opt,
                        contest_err: None,
                        problem_errs: problem_errs_opt,
                    }
                }
                Fail(err) => ContestResult {
                    contest: None,
                    contest_err: Some(err),
                    problem_errs: None,
                },
            }
        }
        Err(err) => ContestResult {
            contest: None,
            contest_err: Some(ContestErr(err.to_string())),
            problem_errs: None,
        },
    }
}

/// Gets a contest from a contest path, problem directory, and template directory.

pub fn fetch_contest(
    contest_path: &Path,
    problem_dir: &Path,
    template_dir: &Path,
) -> ContestResult {
    match fs::read_to_string(contest_path) {
        Ok(contest_yaml) => parse_contest_yaml(&contest_yaml, problem_dir, template_dir),
        Err(_) => ContestResult {
            contest: None,
            contest_err: Some(ContestErr(format!(
                "Could not read contest from {:?}",
                contest_path,
            ))),
            problem_errs: None,
        },
    }
}

impl Contest {
    /// Check whether the contest has the proper vars defined, and each problem has the proper problemvars and solutionvars defined

    pub fn check_template(&self) -> Option<Vec<MapmErr>> {
        let mut mapm_errs: Vec<MapmErr> = Vec::new();

        for var in &self.template.1.vars {
            if !self.vars.contains_key(var) {
                mapm_errs.push(ContestErr(format!("Does not contain key `{}`", &var)));
            }
        }
        if mapm_errs.len() > 0 {
            Some(mapm_errs)
        } else {
            None
        }
    }
    /// Compiles the contest given a template directory to query

    pub fn compile(&self, template_dir: &Path) -> (Option<MapmErr>, Vec<Result<String, String>>) {
        let cwd = env::current_dir().unwrap();
        let mut results: Vec<Result<String, String>> = Vec::new();

        let template_path = &template_dir.join(&self.template.0);
        let build_dir = Path::new("build");
        if !build_dir.is_dir() {
            // If `build` exists but is not a directory, user intervention is required - so we panic.
            if build_dir.exists() {
                panic!(
                    "{:?} exists in {:?} but is not a directory; please move it to fix this issue.",
                    build_dir, cwd,
                );
            }
            fs::create_dir(&build_dir).expect(&format!(
                "Could not create directory {:?} in {:?}",
                build_dir, cwd,
            ));
        }

        if !template_path.exists() {
            return (
                Some(TemplateErr(format!(
                    "Could not read template {:?}",
                    template_path
                ))),
                results,
            );
        }

        copy_dir_ignore_dots(template_path, Path::new("build")).unwrap();

        env::set_current_dir(&build_dir).expect(&format!(
            "Could not set current directory to {:?}",
            &build_dir
        ));

        let mut headers = String::new();

        for (key, val) in &self.vars {
            headers.push_str(
                &[
                    "\\expandafter\\def\\csname mapm@var@",
                    &key,
                    "\\endcsname{",
                    &val,
                    "}\n",
                ]
                .concat(),
            );
        }

        headers.push_str(&write_as_tex(
            self.problems.iter().map(|p| p.1.clone()).collect(),
        ));

        fs::write("mapm-headers.tex", headers).expect(&format!(
            "Could not write `mapm-headers.tex` to directory {:?}",
            template_dir,
        ));

        for (tex_path, unparsed_pdf_path) in &self.template.1.texfiles {
            let left_braces: Vec<usize> = unparsed_pdf_path
                .match_indices("{")
                .map(|(i, _)| i)
                .collect();
            let right_braces: Vec<usize> = unparsed_pdf_path
                .match_indices("}")
                .map(|(i, _)| i)
                .collect();
            if left_braces.len() != right_braces.len() {
                return (
                    Some(TemplateErr(format!(
                        "Unbalanced braces for the value `{}` of key `{}` in the `texfiles` of template `{}`",
                        unparsed_pdf_path,
                        tex_path,
                        &self.template.0,
                    ))),
                    results,
                );
            }

            let mut parsed_pdf_path = String::new();

            // We reuse this errmsg, so define it here
            let nested_braces_error = format!("Nested braces are not supported.\nSee the value `{}` of key `{}` in the `texfiles` of template `{}`", unparsed_pdf_path, tex_path, &self.template.0);

            if left_braces.len() > 0 {
                for n in 0..left_braces.len() {
                    if left_braces[n] > right_braces[n] {
                        return (Some(TemplateErr(nested_braces_error)), results);
                    }
                }

                for n in 0..left_braces.len() - 1 {
                    if right_braces[n] > left_braces[n + 1] {
                        return (Some(TemplateErr(nested_braces_error)), results);
                    }
                }

                for n in 0..left_braces.len() {
                    let key = &unparsed_pdf_path[left_braces[n] + 1..right_braces[n]];
                    match self.vars.get(key) {
                        Some(val) => {
                            if n == 0 {
                                parsed_pdf_path.push_str(&unparsed_pdf_path[..left_braces[n]]);
                            } else {
                                parsed_pdf_path.push_str(
                                    &unparsed_pdf_path[right_braces[n - 1] + 1..left_braces[n]],
                                );
                            }
                            parsed_pdf_path.push_str(val);
                        }
                        None => {
                            return (
                                Some(TemplateErr(
                                    [
                                        "The value `",
                                        unparsed_pdf_path,
                                        "` of key `",
                                        tex_path,
                                        "` in the `texfiles` of template `",
                                        &self.template.0,
                                        "`\nis referencing undefined template variable `",
                                        key,
                                        "`",
                                    ]
                                    .concat(),
                                )),
                                results,
                            );
                        }
                    }
                }

                parsed_pdf_path
                    .push_str(&unparsed_pdf_path[right_braces[right_braces.len() - 1] + 1..]);
            } else {
                parsed_pdf_path.push_str(&unparsed_pdf_path);
            }
            let latexmk = Command::new("latexmk").args([&format!("-pdflatex={}", &self.template.1.engine), "-pdf", "-f", tex_path]).output().expect("Failed to execute latexmk.\nMake sure you have the latexmk script installed\nand make sure it is in your PATH.");
            if latexmk.status.success() {
                results.push(Ok(format!("Finished compiling `{}`", &parsed_pdf_path)));
                let original_pdf_path = Path::new(tex_path).with_extension("pdf");

                fs::rename(&original_pdf_path, &cwd.join(&parsed_pdf_path)).expect(&format!(
                    "Could not rename {:?} to `{}`",
                    &original_pdf_path, &parsed_pdf_path
                ));
            } else {
                results.push(Err(format!("Failed to compile {}", &parsed_pdf_path)));
            }
        }

        env::set_current_dir(&cwd).expect(&format!(
            "Failed to set current working directory to {:?}",
            cwd
        ));

        (None, results)
    }
}
