use std::collections::HashMap;

use crate::result::MapmErr::*;
use crate::result::MapmResult;
use crate::result::MapmResult::*;

use std::fs;
use std::path::Path;

use serde::{Deserialize, Serialize};

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct Template {
    pub engine: String,
    pub texfiles: HashMap<String, String>,
    pub problem_count: u32,
    pub vars: Vec<String>,
    pub problemvars: Vec<String>,
    pub solutionvars: Vec<String>,
}

/// Gets a template config from `~/.mapm/template` based on name of template (which corresponds to filepath)

pub fn fetch_template_config(template_name: &str, templates_dir: &Path) -> MapmResult<Template> {
    let template_path = templates_dir.join(template_name).join("config.yml");
    match fs::read_to_string(&template_path) {
        Ok(template_yaml) => parse_template_yaml(&template_yaml),
        Err(_) => Fail(TemplateErr(
            [
                "Could not read template ",
                &template_name,
                " from `",
                &template_path.to_str().unwrap(),
                "`",
            ]
            .concat(),
        )),
    }
}

/// Parses template yaml into a Template
///
/// # Usage
///
/// ```
/// use mapm::template::Template;
/// use mapm::template::parse_template_yaml;
/// use std::collections::HashMap;
///
/// let template_yaml = "engine: pdflatex
/// texfiles:
///   problems.tex: ${title}.PDF
///   solutions.tex: ${title}-sols.PDF
/// problem_count: 1
/// vars:
///   - title
///   - year
/// problemvars:
///   - problem
/// solutionvars:
///   - text
///   - author
/// ";
///
/// let template = parse_template_yaml(template_yaml).unwrap();
///
/// let mut texfiles = HashMap::new();
/// texfiles.insert(String::from("problems.tex"), String::from("${title}.PDF"));
/// texfiles.insert(String::from("solutions.tex"), String::from("${title}-sols.PDF"));
///
/// assert_eq!(template.engine, "pdflatex");
/// assert_eq!(template.texfiles, texfiles);
/// assert_eq!(template.vars, Vec::from([String::from("title"), String::from("year")]));
/// assert_eq!(template.problemvars, Vec::from([String::from("problem")]));
/// assert_eq!(template.solutionvars, Vec::from([String::from("text"), String::from("author")]));
/// ```

pub fn parse_template_yaml(yaml: &str) -> MapmResult<Template> {
    match serde_yaml::from_str::<Template>(yaml) {
        Ok(template) => Success(template),
        Err(err) => Fail(TemplateErr(err.to_string())),
    }
}
