use linked_hash_map::LinkedHashMap;

use crate::mapm_result::MapmErr::*;
use crate::mapm_result::MapmResult;
use crate::mapm_result::MapmResult::*;

use strict_yaml_rust::StrictYaml;
use strict_yaml_rust::StrictYamlLoader;

pub struct Template {
    pub engine: String,
    pub texfiles: LinkedHashMap<String, String>,
    pub vars: Vec<String>,
    pub problemvars: Vec<String>,
    pub solutionvars: Vec<String>,
}

/// Parses template yaml into a Template
///
/// # Usage
///
/// ```
/// use mapm::template::Template;
/// use mapm::template::parse_template_yaml;
/// use linked_hash_map::LinkedHashMap;
///
/// let template_yaml = "engine: pdflatex
/// texfiles:
///   problems.tex: ${title}.PDF
///   solutions.tex: ${title}-sols.PDF
/// vars:
///   - title
///   - year
/// problemvars:
///   - problem
/// solutionvars:
///   - text
///   - author
/// ";
///
/// let template = parse_template_yaml(template_yaml).unwrap();
///
/// let mut texfiles = LinkedHashMap::new();
/// texfiles.insert(String::from("problems.tex"), String::from("${title}.PDF"));
/// texfiles.insert(String::from("solutions.tex"), String::from("${title}-sols.PDF"));
///
/// assert_eq!(template.engine, "pdflatex");
/// assert_eq!(template.texfiles, texfiles);
/// assert_eq!(template.vars, Vec::from([String::from("title"), String::from("year")]));
/// assert_eq!(template.problemvars, Vec::from([String::from("problem")]));
/// assert_eq!(template.solutionvars, Vec::from([String::from("text"), String::from("author")]));
/// ```

pub fn parse_template_yaml(yaml: &str) -> MapmResult<Template> {
    let parsed_yaml = StrictYamlLoader::load_from_str(yaml);
    match parsed_yaml {
        Ok(data) => {
            let mut engine: String = String::new();
            let mut texfiles: LinkedHashMap<String, String> = LinkedHashMap::new();
            let mut vars: Vec<String> = Vec::new();
            let mut problemvars: Vec<String> = Vec::new();
            let mut solutionvars: Vec<String> = Vec::new();

            let map = data[0].as_hash();
            match map {
                Some(map) => {
                    match map.get(&StrictYaml::from_str("engine")) {
                        Some(engine_strict_yaml) => match engine_strict_yaml.as_str() {
                            Some(engine_str) => {
                                engine.push_str(engine_str);
                            }
                            None => {
                                return Fail(TemplateErr(String::from(
                                    "Could not parse value of key `engine` in template yaml",
                                )));
                            }
                        },
                        None => {
                            return Fail(TemplateErr(String::from(
                                "Key `engine` is not present in template yaml",
                            )));
                        }
                    }
                    match map.get(&StrictYaml::from_str("texfiles")) {
                        Some(texfiles_strict_yaml) => match texfiles_strict_yaml.as_hash() {
                            Some(texfiles_hash) => {
                                for (key, val) in texfiles_hash {
                                    texfiles.insert(
                                        String::from(key.as_str().unwrap()),
                                        String::from(val.as_str().unwrap()),
                                    );
                                }
                            }
                            None => {
                                return Fail(TemplateErr(String::from(
                                    "Could not parse value of key `texfiles` in template yaml",
                                )));
                            }
                        },
                        None => {
                            return Fail(TemplateErr(String::from(
                                "Key `texfiles` is not present in template yaml",
                            )));
                        }
                    }
                    match map.get(&StrictYaml::from_str("vars")) {
                        Some(vars_strict_yaml) => match vars_strict_yaml.as_vec() {
                            Some(vars_vec) => {
                                for val in vars_vec {
                                    vars.push(String::from(val.as_str().unwrap()));
                                }
                            }
                            None => {
                                return Fail(TemplateErr(String::from(
                                    "Could not parse value of key `vars` in template yaml",
                                )));
                            }
                        },
                        None => {
                            return Fail(TemplateErr(String::from(
                                "Key `vars` is not present in template yaml",
                            )));
                        }
                    }
                    match map.get(&StrictYaml::from_str("problemvars")) {
                        Some(problemvars_strict_yaml) => match problemvars_strict_yaml.as_vec() {
                            Some(problemvars_vec) => {
                                for val in problemvars_vec {
                                    problemvars.push(String::from(val.as_str().unwrap()));
                                }
                            }
                            None => {
                                return Fail(TemplateErr(String::from(
                                    "Could not parse value of key `problemvars` in template yaml",
                                )));
                            }
                        },
                        None => {
                            return Fail(TemplateErr(String::from(
                                "Key `problemvars` is not present in template yaml",
                            )));
                        }
                    }
                    match map.get(&StrictYaml::from_str("solutionvars")) {
                        Some(solutionvars_strict_yaml) => match solutionvars_strict_yaml.as_vec() {
                            Some(solutionvars_vec) => {
                                for val in solutionvars_vec {
                                    solutionvars.push(String::from(val.as_str().unwrap()));
                                }
                            }
                            None => {
                                return Fail(TemplateErr(String::from(
                                    "Could not parse value of key `solutionvars` in template yaml",
                                )));
                            }
                        },
                        None => {
                            return Fail(TemplateErr(String::from(
                                "Key `solutionvars` is not present in template yaml",
                            )));
                        }
                    }
                }
                None => {
                    return Fail(TemplateErr(String::from("Template yaml is empty")));
                }
            }

            let template = Template {
                engine,
                texfiles,
                vars,
                problemvars,
                solutionvars,
            };

            return Success(template);
        }
        Err(e) => {
            return Fail(TemplateErr(
                ["Could not parse template yaml: \n", &e.to_string()].concat(),
            ));
        }
    }
}
