//! Parses and manipulates problem yaml

use crate::mapm_result::MapmErr;
use crate::mapm_result::MapmErr::*;
use crate::mapm_result::MapmResult;
use crate::mapm_result::MapmResult::*;

use crate::template::Template;
use Filter::*;
use FilterAction::*;
use Views::*;

use linked_hash_map::LinkedHashMap;

use strict_yaml_rust::StrictYamlLoader;

pub type Vars = LinkedHashMap<String, String>;
pub type Solutions = Vec<LinkedHashMap<String, String>>;

pub struct Problem {
    pub vars: Vars,
    pub solutions: Solutions,
}

/// Defines types of filters for problems
///
/// Only the "Exists" variant may be used for the special key "solutions".
///
/// For the Gt, Lt, Ge, Le conditions, whoever is working with the problem files (either through CLI utils or a GUI) is responsible for ensuring that any key being compared with Gt, Lt, Ge, Le is a non-negative integer

pub enum Filter {
    Exists { key: String },
    Eq { key: String, val: String },
    Gt { key: String, val: u32 },
    Lt { key: String, val: u32 },
    Ge { key: String, val: u32 },
    Le { key: String, val: u32 },
}

/// Defines the actions for filtering problems
///
/// Positive means "keep if this condition is satisfied", negative means "keep if this condition is **not** satisfied"

pub enum FilterAction {
    Positive(Filter),
    Negative(Filter),
}

/// Views *only* shown strings or views everything *but* hidden strings
pub enum Views {
    Show(Vec<String>),
    Hide(Vec<String>),
}

impl Problem {
    /// Converts Problem into TeX string to be inputted in `mapm-headers.tex`
    ///
    /// # Usage
    ///
    /// ```
    /// use mapm::problem::Problem;
    /// use linked_hash_map::LinkedHashMap;
    ///
    /// let mut vars = LinkedHashMap::new();
    /// vars.insert(String::from("problem"), String::from("What is $1+1$?"));
    /// vars.insert(String::from("author"), String::from("Dennis Chen"));
    ///
    /// let mut solution_one = LinkedHashMap::new();
    /// solution_one.insert(String::from("text"), String::from("It's probably $2$."));
    /// solution_one.insert(String::from("author"), String::from("Dennis Chen"));
    /// let mut solution_two = LinkedHashMap::new();
    /// solution_two.insert(String::from("text"), String::from("The answer is $2$, but my proof is too small to fit into the margin."));
    /// solution_two.insert(String::from("author"), String::from("Pierre de Fermat"));
    ///
    /// let mut solutions = Vec::from([solution_one, solution_two]);
    ///
    /// let problem = Problem { vars, solutions };
    ///
    /// let problem_tex = "\\def\\mapm@probvar@1@problem{What is $1+1$?}\n\\def\\mapm@probvar@1@author{Dennis Chen}\n\\def\\mapm@solcount@1{2}\\def\\mapm@solvar@1@1@text{It's probably $2$.}\n\\def\\mapm@solvar@1@1@author{Dennis Chen}\n\\def\\mapm@solvar@1@2@text{The answer is $2$, but my proof is too small to fit into the margin.}\n\\def\\mapm@solvar@1@2@author{Pierre de Fermat}\n";
    /// assert_eq!(problem_tex, problem.as_tex(1));
    /// ```
    pub fn as_tex(&self, problem_number: u32) -> String {
        let mut tex: String = String::new();
        for (key, val) in &self.vars {
            tex.push_str(
                &[
                    "\\expandafter\\def\\csname mapm@probvar@",
                    &problem_number.to_string(),
                    "@",
                    &key,
                    "\\endcsname{",
                    &val,
                    "}\n",
                ]
                .concat(),
            );
        }
        tex.push_str(
            &[
                "\\expandafter\\def\\csname mapm@solcount@",
                &problem_number.to_string(),
                "\\endcsname{",
                &self.solutions.len().to_string(),
                "}",
            ]
            .concat(),
        );
        for (pos, map) in self.solutions.iter().enumerate() {
            for (key, val) in map {
                let solution_number = (&pos + 1).to_string();
                tex.push_str(
                    &[
                        "\\expandafter\\def\\csname mapm@solvar@",
                        &problem_number.to_string(),
                        "@",
                        &solution_number,
                        "@",
                        &key,
                        "\\endcsname{",
                        &val,
                        "}\n",
                    ]
                    .concat(),
                );
            }
        }
        return tex;
    }

    /// Converts Problem into yaml, may be useful when some interface needs to directly modify a problem (like in a GUI app)
    ///
    /// The function uses two spaces for yaml indents, not four
    ///
    /// # Usage
    ///
    /// ```
    /// use mapm::problem::Problem;
    /// use linked_hash_map::LinkedHashMap;
    ///
    /// let mut vars = LinkedHashMap::new();
    /// vars.insert(String::from("problem"), String::from("What is $1+1$?"));
    /// vars.insert(String::from("author"), String::from("Dennis Chen"));
    ///
    /// let mut solution_one = LinkedHashMap::new();
    /// solution_one.insert(String::from("text"), String::from("It's probably $2$."));
    /// solution_one.insert(String::from("author"), String::from("Dennis Chen"));
    /// let mut solution_two = LinkedHashMap::new();
    /// solution_two.insert(String::from("text"), String::from("The answer is $2$, but my proof is too small to fit into the margin."));
    /// solution_two.insert(String::from("author"), String::from("Pierre de Fermat"));
    ///
    /// let mut solutions = Vec::from([solution_one, solution_two]);
    ///
    /// let problem = Problem { vars, solutions };
    ///
    /// let yaml = "problem: What is $1+1$?
    /// author: Dennis Chen
    /// solutions:
    ///   - text: It's probably $2$.
    ///     author: Dennis Chen
    ///   - text: The answer is $2$, but my proof is too small to fit into the margin.
    ///     author: Pierre de Fermat
    /// ";
    ///
    /// assert_eq!(problem.as_yaml(), yaml);
    /// ```
    pub fn as_yaml(&self) -> String {
        let mut yaml: String = String::new();
        for (key, val) in &self.vars {
            yaml.push_str(&[&key, ": ", &val, "\n"].concat());
        }
        yaml.push_str("solutions:\n");
        for map in &self.solutions {
            let mut first: bool = true;
            for (key, val) in map {
                if first {
                    yaml.push_str(&["  - ", &key, ": ", &val, "\n"].concat());
                    first = false;
                } else {
                    yaml.push_str(&["    ", &key, ": ", &val, "\n"].concat());
                }
            }
        }
        return yaml;
    }

    /// Checks if a problem passes or fails a certain filter
    ///
    /// For Gt, it checks if the value of the problem is greater than the value passed in the filter (not the other way around). The same is true of Lt, Ge, Le.
    ///
    /// # Usage
    ///
    /// ```
    /// use mapm::problem::Problem;
    /// use mapm::problem::Filter;
    /// use linked_hash_map::LinkedHashMap;
    ///
    /// let mut vars = LinkedHashMap::new();
    /// vars.insert(String::from("problem"), String::from("What is $1+1$?"));
    /// vars.insert(String::from("author"), String::from("Dennis Chen"));
    /// vars.insert(String::from("difficulty"), String::from("5"));
    ///
    /// let mut solution_one = LinkedHashMap::new();
    /// solution_one.insert(String::from("text"), String::from("It's probably $2$."));
    /// solution_one.insert(String::from("author"), String::from("Dennis Chen"));
    /// let mut solution_two = LinkedHashMap::new();
    /// solution_two.insert(String::from("text"), String::from("The answer is $2$, but my proof is too small to fit into the margin."));
    /// solution_two.insert(String::from("author"), String::from("Pierre de Fermat"));
    ///
    /// let mut solutions = Vec::from([solution_one, solution_two]);
    ///
    /// let problem = Problem { vars, solutions };
    ///
    /// assert_eq!(problem.try_filter(Filter::Exists{key: String::from("subject")}), false);
    /// assert_eq!(problem.try_filter(Filter::Exists{key: String::from("solutions")}), true);
    ///
    /// assert_eq!(problem.try_filter(Filter::Gt{key: String::from("difficulty"), val: 5}), false);
    /// assert_eq!(problem.try_filter(Filter::Ge{key: String::from("difficulty"), val: 5}), true);
    /// assert_eq!(problem.try_filter(Filter::Lt{key: String::from("difficulty"), val: 5}), false);
    /// assert_eq!(problem.try_filter(Filter::Le{key: String::from("difficulty"), val: 5}), true);
    ///
    /// ```

    pub fn try_filter(&self, filter: Filter) -> bool {
        match filter {
            Exists { key } => {
                if key == "solutions" {
                    return self.solutions.len() > 0;
                } else {
                    return self.vars.contains_key(&key);
                }
            }
            Eq { key, val } => match self.vars.get(&key) {
                Some(problem_value) => {
                    return problem_value == &val;
                }
                None => {
                    return false;
                }
            },
            Gt { key, val } => match self.vars.get(&key) {
                Some(problem_value) => match problem_value.parse::<u32>() {
                    Ok(problem_value_u32) => {
                        return problem_value_u32 > val;
                    }
                    Err(_) => return false,
                },
                None => {
                    return false;
                }
            },
            Lt { key, val } => match self.vars.get(&key) {
                Some(problem_value) => match problem_value.parse::<u32>() {
                    Ok(problem_value_u32) => {
                        return problem_value_u32 < val;
                    }
                    Err(_) => return false,
                },
                None => {
                    return false;
                }
            },
            Ge { key, val } => match self.vars.get(&key) {
                Some(problem_value) => match problem_value.parse::<u32>() {
                    Ok(problem_value_u32) => {
                        return problem_value_u32 >= val;
                    }
                    Err(_) => return false,
                },
                None => {
                    return false;
                }
            },
            Le { key, val } => match self.vars.get(&key) {
                Some(problem_value) => match problem_value.parse::<u32>() {
                    Ok(problem_value_u32) => {
                        return problem_value_u32 <= val;
                    }
                    Err(_) => return false,
                },
                None => {
                    return false;
                }
            },
        }
    }

    /// Checks whether a problem satisfies a set of filters, and if it does, return it; otherwise return `None`
    ///
    /// If `filters` is empty, then every problem will pass the filter.

    pub fn filter(self, filters: Vec<FilterAction>) -> Option<Problem> {
        for filter_action in filters {
            match filter_action {
                Positive(filter) => {
                    if self.try_filter(filter) {
                        return Some(self);
                    } else {
                        return None;
                    }
                }
                Negative(filter) => {
                    if self.try_filter(filter) {
                        return None;
                    } else {
                        return Some(self);
                    }
                }
            }
        }
        return Some(self);
    }

    /// Show only the keys passed in or everything but the keys passed in to a problem
    ///
    /// If `views` is empty, keeps every key by default (does nothing)

    pub fn filter_keys(self, views: Views) -> (Vars, Option<Solutions>) {
        match views {
            Show(tags) => {
                let mut vars: Vars = LinkedHashMap::new();
                let mut solutions: bool = false;
                for (key, val) in self.vars {
                    if key == "solutions" {
                        solutions = true;
                    } else {
                        if tags.contains(&key) {
                            vars.insert(key, val);
                        }
                    }
                }
                if solutions {
                    return (vars, Some(self.solutions));
                } else {
                    return (vars, None);
                }
            }
            Hide(tags) => {
                let mut vars: Vars = LinkedHashMap::new();
                let mut solutions: bool = true;
                for (key, val) in self.vars {
                    if key == "solutions" {
                        solutions = false;
                    } else {
                        if !tags.contains(&key) {
                            vars.insert(key, val);
                        }
                    }
                }
                if solutions {
                    return (vars, None);
                } else {
                    return (vars, Some(self.solutions));
                }
            }
        }
    }

    /// Checks if a problem contains all the variables in a template
    ///
    /// Returns None if the problem successfully passes the test, returns Some(MapmErr) otherwise
    ///
    /// # Usage
    ///
    /// ## Expected success
    ///
    /// ```
    /// use mapm::problem::*;
    /// use mapm::template::*;
    /// let problem_yaml = "problem: What is $1+1$?
    /// solutions:
    ///   - text: Some say the answer is $2$.
    ///     author: Dennis Chen
    /// ";
    ///
    /// let template_yaml = "engine: pdflatex
    /// texfiles:
    ///   problems.tex: ${title}.PDF
    ///   solutions.tex: ${title}-sols.PDF
    /// vars:
    ///   - title
    ///   - year
    /// problemvars:
    ///   - problem
    /// solutionvars:
    ///   - text
    ///   - author
    /// ";
    ///
    /// let problem = parse_problem_yaml(problem_yaml).unwrap();
    /// let template = parse_template_yaml(template_yaml).unwrap();
    ///
    /// assert!(problem.check_template(template).is_none());
    /// ```
    ///
    /// ## Expected failure
    ///
    /// ```
    /// use mapm::problem::*;
    /// use mapm::template::*;
    /// use mapm::mapm_result::MapmErr;
    /// use mapm::mapm_result::MapmErr::*;
    /// let problem_yaml = "problem: What is $1+1$?
    /// solutions:
    ///   - text: Some say the answer is $2$.
    ///     author: Dennis Chen
    /// ";
    ///
    /// let template_yaml = "engine: pdflatex
    /// texfiles:
    ///   problems.tex: ${title}.PDF
    ///   solutions.tex: ${title}-sols.PDF
    /// vars:
    ///   - title
    ///   - year
    /// problemvars:
    ///   - problem
    ///   - author
    /// solutionvars:
    ///   - text
    ///   - author
    /// ";
    ///
    /// let problem = parse_problem_yaml(problem_yaml).unwrap();
    /// let template = parse_template_yaml(template_yaml).unwrap();
    ///
    /// let template_check = problem.check_template(template).unwrap();
    ///
    /// assert_eq!(template_check.len(), 1);
    /// match &template_check[0] {
    ///     ProblemErr(err) => {
    ///         assert_eq!(err, "Does not contain key `author`");
    ///     }
    ///     _ => {
    ///         panic!("MapmErr type is not ProblemErr");
    ///     }
    /// }
    /// ```

    pub fn check_template(&self, template: &Template) -> Option<Vec<MapmErr>> {
        let mut mapm_errs: Vec<MapmErr> = Vec::new();

        for problemvar in &template.problemvars {
            if !self.vars.contains_key(problemvar) {
                mapm_errs.push(ProblemErr(
                    ["Does not contain key `", &problemvar, "`"].concat(),
                ));
            }
        }
        let mut index: u32 = 0;
        for map in &self.solutions {
            index += 1;
            for solutionvar in &template.solutionvars {
                if !map.contains_key(solutionvar) {
                    mapm_errs.push(SolutionErr(
                        [
                            "Solution `",
                            &index.to_string(),
                            "` does not contain key `",
                            &solutionvar,
                            "`",
                        ]
                        .concat(),
                    ));
                }
            }
        }
        if mapm_errs.len() > 0 {
            return Some(mapm_errs);
        } else {
            return None;
        }
    }
}

///
/// # Usage
///
/// ```
/// use mapm::problem::Problem;
/// use mapm::problem::parse_problem_yaml;
/// use linked_hash_map::LinkedHashMap;
/// let yaml = "
/// problem: What is $1+1$?
/// author: Dennis Chen
/// solutions:
///   - text: It's probably $2$.
///     author: Dennis Chen
///   - text: The answer is $2$, but my proof is too small to fit into the margin.
///     author: Pierre de Fermat
/// ";
/// let problem = parse_problem_yaml(yaml).unwrap();
///
/// let mut vars = LinkedHashMap::new();
/// vars.insert(String::from("problem"), String::from("What is $1+1$?"));
/// vars.insert(String::from("author"), String::from("Dennis Chen"));
///
/// let mut solution_one = LinkedHashMap::new();
/// solution_one.insert(String::from("text"), String::from("It's probably $2$."));
/// solution_one.insert(String::from("author"), String::from("Dennis Chen"));
/// let mut solution_two = LinkedHashMap::new();
/// solution_two.insert(String::from("text"), String::from("The answer is $2$, but my proof is too small to fit into the margin."));
/// solution_two.insert(String::from("author"), String::from("Pierre de Fermat"));
///
/// let mut solutions = Vec::from([solution_one, solution_two]);
///
/// assert_eq!(problem.vars, vars);
/// assert_eq!(problem.solutions, solutions);
/// ```

pub fn parse_problem_yaml(yaml: &str) -> MapmResult<Problem> {
    let parsed_yaml = StrictYamlLoader::load_from_str(yaml);
    match parsed_yaml {
        Ok(data) => {
            let mut vars: Vars = LinkedHashMap::new();
            let mut solutions: Solutions = Vec::new();

            let map = data[0].as_hash();
            match map {
                Some(map) => {
                    for (key, val) in map {
                        if key.as_str().unwrap() == "solutions" {
                            match val.as_vec() {
                                Some(solutions_vec) => {
                                    for solution_hash in solutions_vec {
                                        match solution_hash.as_hash() {
                                            Some(solution_hash) => {
                                                let mut solution: LinkedHashMap<String, String> =
                                                    LinkedHashMap::new();
                                                for (key, val) in solution_hash {
                                                    solution.insert(
                                                        String::from(key.as_str().unwrap()),
                                                        String::from(val.as_str().unwrap()),
                                                    );
                                                }
                                                solutions.push(solution);
                                            }
                                            None => {
                                                return Fail(ProblemErr(String::from("Some element in 'solutions' has no keys, or they couldn't be interpreted")));
                                            }
                                        }
                                    }
                                }
                                None => {
                                    return Fail(SolutionErr(String::from("No array 'solutions' in yaml, or could not interpret 'solutions' as an array")));
                                }
                            }
                        } else {
                            vars.insert(
                                String::from(key.as_str().unwrap()),
                                String::from(val.as_str().unwrap()),
                            );
                        }
                    }

                    return Success(Problem { vars, solutions });
                }
                None => {
                    return Fail(ProblemErr(String::from("Problem yaml is empty")));
                }
            }
        }
        Err(e) => {
            return Fail(ProblemErr(
                ["Could not parse problem yaml:\n", &e.to_string()].concat(),
            ));
        }
    }
}
