//! Provides functions for file i/o

use std::env;
use std::fs;
use std::path::Path;
use std::process::Command;

use crate::problem::parse_problem_yaml;
use crate::problem::Problem;

use crate::template::parse_template_yaml;
use crate::template::Template;

use crate::contest::*;

use crate::mapm_result::MapmErr;
use crate::mapm_result::MapmErr::*;
use crate::mapm_result::MapmResult;
use crate::mapm_result::MapmResult::*;

use fs_extra;

/// Gets a problem from `~/.mapm/problems` based on name of problem (which corresponds to filepath)

pub fn fetch_problem(problem_name: &str) -> MapmResult<Problem> {
    let home = env::var("HOME").unwrap();
    match fs::read_to_string(
        Path::new(&home)
            .join(".mapm")
            .join("problems")
            .join(&[problem_name, ".yml"].concat()),
    ) {
        Ok(problem_yaml) => match parse_problem_yaml(&problem_yaml) {
            Success(problem) => {
                return Success(problem);
            }
            err => {
                return err;
            }
        },
        Err(_) => {
            return Fail(ProblemErr(
                [
                    "Could not read problem ",
                    &problem_name,
                    " from `",
                    &home,
                    "/.mapm/",
                    &problem_name,
                    ".yml`",
                ]
                .concat(),
            ));
        }
    }
}

/// Gets a template config from `~/.mapm/template` based on name of template (which corresponds to filepath)

pub fn fetch_template_config(template_name: &str) -> MapmResult<Template> {
    let home = env::var("HOME").unwrap();
    let template_path = Path::new(&home)
        .join(".mapm")
        .join("templates")
        .join(template_name)
        .join("config.yml");
    match fs::read_to_string(&template_path) {
        Ok(template_yaml) => match parse_template_yaml(&template_yaml) {
            Success(template) => {
                return Success(template);
            }
            err => {
                return err;
            }
        },
        Err(_) => {
            return Fail(TemplateErr(
                [
                    "Could not read template ",
                    &template_name,
                    " from `",
                    &template_path.to_str().unwrap(),
                    "`",
                ]
                .concat(),
            ));
        }
    }
}

/// Gets a contest from a given path

pub fn fetch_contest(contest_path: &Path) -> ContestResult {
    match fs::read_to_string(contest_path) {
        Ok(contest_yaml) => {
            return parse_contest_yaml(&contest_yaml);
        }
        Err(_) => {
            return ContestResult {
                contest: None,
                contest_errs: Some(Vec::from([ContestErr(
                    [
                        "Could not read contest from `",
                        contest_path.to_str().unwrap(),
                        "`",
                    ]
                    .concat(),
                )])),
                problem_errs: None,
            };
        }
    }
}

/// Compiles a contest

pub fn compile_contest(contest: &Contest) -> (Option<MapmErr>, Vec<Result<String, String>>) {
    let home = env::var("HOME").unwrap();
    let cwd = env::current_dir().unwrap();
    let mut results: Vec<Result<String, String>> = Vec::new();

    let template_dir = Path::new(&home)
        .join(".mapm")
        .join("templates")
        .join(&contest.template.0);
    let build_dir = Path::new("build");
    fs::create_dir(&build_dir).ok();

    if !template_dir.exists() {
        return (
            Some(TemplateErr(
                [
                    "Could not read template `",
                    &template_dir.to_str().unwrap(),
                    "`",
                ]
                .concat(),
            )),
            results,
        );
    }
    let options = fs_extra::dir::CopyOptions {
        overwrite: true,
        skip_exist: false,
        buffer_size: 64000,
        copy_inside: false,
        content_only: true,
        depth: 0,
    };
    fs_extra::dir::copy(template_dir, "build", &options).unwrap();

    env::set_current_dir(&build_dir).ok();

    fs::write("mapm-headers.tex", contest.as_tex()).ok();

    for (tex_path, unparsed_pdf_path) in &contest.template.1.texfiles {
        let left_braces: Vec<usize> = unparsed_pdf_path
            .match_indices("{")
            .map(|(i, _)| i)
            .collect();
        let right_braces: Vec<usize> = unparsed_pdf_path
            .match_indices("}")
            .map(|(i, _)| i)
            .collect();
        if left_braces.len() != right_braces.len() {
            return (
                Some(TemplateErr(
                    [
                        "Unbalanced braces for the value `",
                        unparsed_pdf_path,
                        "` of key `",
                        tex_path,
                        "` in the `texfiles` of template `",
                        &contest.template.0,
                        "`",
                    ]
                    .concat(),
                )),
                results,
            );
        }

        for n in 0..left_braces.len() {
            if left_braces[n] > right_braces[n] {
                return (
                    Some(TemplateErr(
                        [
                            "Nested braces are not supported.\nSee the value `",
                            unparsed_pdf_path,
                            "` of key `",
                            tex_path,
                            "` in the `texfiles` of template `",
                            &contest.template.0,
                            "`",
                        ]
                        .concat(),
                    )),
                    results,
                );
            }
        }

        for n in 0..left_braces.len() - 1 {
            if right_braces[n] > left_braces[n + 1] {
                return (
                    Some(TemplateErr(
                        [
                            "Nested braces are not supported.\nSee the value `",
                            unparsed_pdf_path,
                            "` of key `",
                            tex_path,
                            "` in the `texfiles` of template `",
                            &contest.template.0,
                            "`",
                        ]
                        .concat(),
                    )),
                    results,
                );
            }
        }

        let mut parsed_pdf_path = String::new();

        for n in 0..left_braces.len() {
            let key = &unparsed_pdf_path[left_braces[n] + 1..right_braces[n]];
            match contest.vars.get(key) {
                Some(val) => {
                    if n == 0 {
                        parsed_pdf_path.push_str(&unparsed_pdf_path[..left_braces[n]]);
                    } else {
                        parsed_pdf_path
                            .push_str(&unparsed_pdf_path[right_braces[n - 1] + 1..left_braces[n]]);
                    }
                    parsed_pdf_path.push_str(val);
                }
                None => {
                    return (
                        Some(TemplateErr(
                            [
                                "The value `",
                                unparsed_pdf_path,
                                "` of key `",
                                tex_path,
                                "` in the `texfiles` of template `",
                                &contest.template.0,
                                "`\nis referencing undefined template variable `",
                                key,
                                "`",
                            ]
                            .concat(),
                        )),
                        results,
                    );
                }
            }
        }

        parsed_pdf_path.push_str(&unparsed_pdf_path[right_braces[right_braces.len() - 1] + 1..]);

        let latexmk = Command::new("latexmk").args(["-pdf", "-f", tex_path]).output().expect("Failed to execute latexmk.\nMake sure you have the latexmk script installed\nand make sure it is in your PATH.");
        if latexmk.status.success() {
            results.push(Ok(["Finished compiling ", &parsed_pdf_path].concat()));
            let original_pdf_path = Path::new(tex_path).with_extension("pdf");

            fs::rename(&original_pdf_path, &cwd.join(&parsed_pdf_path)).ok();
        } else {
            results.push(Err(["Failed to compile ", &parsed_pdf_path].concat()));
        }
    }

    env::set_current_dir(&cwd).ok();

    return (None, results);
}
