//! Parses and manipulates contest yaml

use linked_hash_map::LinkedHashMap;

use crate::problem;
use crate::problem::*;

use crate::mapm_result::MapmErr;
use crate::mapm_result::MapmErr::*;
use crate::mapm_result::MapmResult::*;

use crate::template::Template;

use crate::file::*;

use strict_yaml_rust::StrictYaml;
use strict_yaml_rust::StrictYamlLoader;

pub struct Contest {
    pub template: (String, Template),
    pub problems: Vec<(String, Problem)>,
    pub vars: LinkedHashMap<String, String>,
}

pub struct ContestResult {
    pub contest: Option<Contest>,
    pub contest_errs: Option<Vec<MapmErr>>,
    pub problem_errs: Option<Vec<(String, Vec<MapmErr>)>>,
}

/// Parses contest yaml into full-fledged contest, reading problem yamls from the corresponding problem names in the contest file
///
/// MapmResult<Contest> will return with a ContestErr if the contest yaml cannot be parsed or does not satisfy the template conditions
///
/// Option<Vec<MapmErr>> will return with Some if any of the problems do not satisfy the template conditions, with the problem name being matched to the errors

pub fn parse_contest_yaml(yaml: &str) -> ContestResult {
    let parsed_yaml = StrictYamlLoader::load_from_str(yaml);
    match parsed_yaml {
        Ok(data) => {
            let map = data[0].as_hash();
            match map {
                Some(map) => {
                    let template_option: Option<Template>;
                    let mut template_string = String::new();
                    let mut vars: problem::Vars = LinkedHashMap::new();
                    let mut problem_names: Vec<String> = Vec::new();
                    let mut contest_errs: Vec<MapmErr> = Vec::new();
                    match map.get(&StrictYaml::from_str("template")) {
                        Some(template_strict_yaml) => match template_strict_yaml.as_str() {
                            Some(template_str) => {
                                template_string.push_str(template_str);
                                match fetch_template_config(template_str) {
                                    Success(template) => {
                                        template_option = Some(template);
                                    }
                                    Fail(err) => {
                                        template_option = None;
                                        contest_errs.push(err);
                                    }
                                }
                            }
                            None => {
                                template_option = None;
                                contest_errs.push(ContestErr(String::from(
                                    "Could not parse value of key `template` in contest yaml",
                                )));
                            }
                        },
                        None => {
                            template_option = None;
                            contest_errs.push(ContestErr(String::from(
                                "Key `template` is not present in contest yaml",
                            )));
                        }
                    }
                    match map.get(&StrictYaml::from_str("vars")) {
                        Some(vars_strict_yaml) => match vars_strict_yaml.as_hash() {
                            Some(vars_hash) => {
                                for (key, val) in vars_hash {
                                    vars.insert(
                                        String::from(key.as_str().unwrap()),
                                        String::from(val.as_str().unwrap()),
                                    );
                                }
                            }
                            None => {
                                contest_errs.push(ContestErr(String::from(
                                    "Could not parse value of key `vars` in contest yaml",
                                )));
                            }
                        },
                        None => {
                            contest_errs.push(ContestErr(String::from(
                                "Key `vars` is not present in contest yaml",
                            )));
                        }
                    }
                    match map.get(&StrictYaml::from_str("problems")) {
                        Some(problems_strict_yaml) => match problems_strict_yaml.as_vec() {
                            Some(problems_vec) => {
                                for key in problems_vec {
                                    problem_names.push(String::from(key.as_str().unwrap()));
                                }
                            }
                            None => {
                                contest_errs.push(ContestErr(String::from(
                                    "Could not parse value of key `problems` in contest yaml",
                                )));
                            }
                        },
                        None => {
                            contest_errs.push(ContestErr(String::from(
                                "Key `problems` is not present in contest yaml",
                            )));
                        }
                    }

                    if contest_errs.len() > 0 {
                        return ContestResult {
                            contest: None,
                            contest_errs: Some(contest_errs),
                            problem_errs: None,
                        };
                    }

                    let template: Template;

                    match template_option {
                        Some(templ) => {
                            template = templ;
                        }
                        None => {
                            return ContestResult {
                                contest: None,
                                contest_errs: Some(contest_errs),
                                problem_errs: None,
                            };
                        }
                    }

                    let mut problems: Vec<(String, Problem)> = Vec::new();
                    let mut problem_errs: Vec<(String, Vec<MapmErr>)> = Vec::new();

                    for problem_name in problem_names {
                        let problem = fetch_problem(&problem_name);
                        match problem {
                            Success(problem) => {
                                problems.push((problem_name, problem));
                            }
                            Fail(err) => {
                                let mut errs = Vec::new();
                                errs.push(err);
                                problem_errs.push((problem_name, errs));
                            }
                        }
                    }
                    for problem in &problems {
                        match problem.1.check_template(&template) {
                            Some(errs) => {
                                problem_errs.push((problem.0.clone(), errs));
                            }
                            None => {}
                        }
                    }

                    let contest = Contest {
                        template: (template_string, template),
                        problems,
                        vars,
                    };

                    let contest_errs_opt: Option<Vec<MapmErr>> = contest.check_template();
                    let problem_errs_opt: Option<Vec<(String, Vec<MapmErr>)>>;

                    if problem_errs.len() == 0 {
                        problem_errs_opt = None;
                    } else {
                        problem_errs_opt = Some(problem_errs);
                    }

                    let contest_opt: Option<Contest>;

                    if matches!(contest_errs_opt, None) && matches!(problem_errs_opt, None) {
                        contest_opt = Some(contest);
                    } else {
                        contest_opt = None;
                    }

                    return ContestResult {
                        contest: contest_opt,
                        contest_errs: contest_errs_opt,
                        problem_errs: problem_errs_opt,
                    };
                }
                None => {
                    return ContestResult {
                        contest: None,
                        contest_errs: Some(Vec::from([ContestErr(String::from(
                            "Contest yaml is empty",
                        ))])),
                        problem_errs: None,
                    };
                }
            }
        }
        Err(e) => {
            return ContestResult {
                contest: None,
                contest_errs: Some(Vec::from([ContestErr(
                    ["Could not parse contest yaml:\n", &e.to_string()].concat(),
                )])),
                problem_errs: None,
            };
        }
    }
}

impl Contest {
    /// Check whether the contest has the proper vars defined, and each problem has the proper problemvars and solutionvars defined

    pub fn check_template(&self) -> Option<Vec<MapmErr>> {
        let mut mapm_errs: Vec<MapmErr> = Vec::new();

        for var in &self.template.1.vars {
            if !self.vars.contains_key(var) {
                mapm_errs.push(ContestErr(["Does not contain key `", &var, "`"].concat()));
            }
        }
        if mapm_errs.len() > 0 {
            return Some(mapm_errs);
        } else {
            return None;
        }
    }
    /// Converts contest into yaml
    pub fn as_yaml(&self) -> String {
        let mut yaml: String = String::new();
        yaml.push_str("template: ");
        yaml.push_str(&self.template.0);
        yaml.push_str("\n");

        yaml.push_str("vars:\n");
        for (key, val) in &self.vars {
            yaml.push_str("  ");
            yaml.push_str(&key);
            yaml.push_str(": ");
            yaml.push_str(&val);
            yaml.push_str("\n");
        }

        yaml.push_str("problems:\n");
        for problem in &self.problems {
            let problem_name = &problem.0;
            yaml.push_str("  - ");
            yaml.push_str(problem_name);
            yaml.push_str("\n");
        }
        return yaml;
    }

    /// Converts contest into tex header
    ///
    /// It does not yield a compilable standalone tex file

    pub fn as_tex(&self) -> String {
        let mut tex: String = String::new();
        for (key, val) in &self.vars {
            tex.push_str(
                &[
                    "\\expandafter\\def\\csname mapm@var@",
                    &key,
                    "\\endcsname{",
                    &val,
                    "}\n",
                ]
                .concat(),
            );
        }
        let mut index = 0;
        for problem in &self.problems {
            index += 1;
            tex.push_str(&problem.1.as_tex(index));
        }
        return tex;
    }
}
