use crate::{msg, templates};
use std::path::Path;

use colour::*;
use mapm::problem::{Solutions, Vars};
use std::process::Command;
use std::{env, fs};

pub fn contest(contest_result: mapm::contest::ContestResult, template_dir: &Path) {
    match contest_result.contest {
        Some(contest) => {
            let contest_msgs = contest.compile(&template_dir);
            match contest_msgs.0 {
                Some(err) => {
                    msg::print_err(&err);
                }
                None => {}
            }
            for result in contest_msgs.1 {
                match result {
                    Ok(msg) => {
                        println!("{}", msg);
                    }
                    Err(err) => {
                        e_red_ln!("{}", err);
                    }
                }
            }
        }
        None => {
            match contest_result.contest_err {
                Some(err) => {
                    msg::print_err(&err);
                    quit::with_code(exitcode::DATAERR);
                }
                None => {}
            }
            match contest_result.problem_errs {
                Some(errs) => {
                    for problem_err in errs {
                        e_red_ln!("Errors for problem `{}`:", &problem_err.0);
                        for err in problem_err.1 {
                            msg::print_err(&err);
                        }
                    }
                }
                None => {}
            }
        }
    }
}

pub fn preview(problem_name: &str, problem_dir: &Path, template_dir: &Path) {
    let contest_yml = &templates::preview_template(problem_name);
    let contest_result =
        mapm::contest::parse_contest_yaml(contest_yml, &problem_dir, &template_dir);
    let cwd = env::current_dir().expect("Could not get current directory");
    let cache_dir = dirs::cache_dir().expect("Could not find cache directory");
    let preview_dir = cache_dir.join("mapm").join(problem_name);
    if !preview_dir.is_dir() {
        fs::create_dir_all(&preview_dir).expect("Could not create directory");
    }
    env::set_current_dir(&preview_dir).expect("Could not set working directory");
    contest(contest_result, &template_dir);
    for pdf in fs::read_dir(&preview_dir).expect("Could not read directory") {
        let path = pdf.unwrap().path();
        match &path.extension() {
            Some(ext) => {
                if ext == &"pdf" {
                    open::that(path).expect("Could not open PDF");
                }
            }
            None => {}
        }
    }
    env::set_current_dir(&cwd).expect("Could not set working directory");
}

pub fn preview_all(problems: Vec<(String, (Vars, Option<Solutions>))>) {
    let problems_tex = templates::preview_all_tex(problems);
    let cwd = env::current_dir().expect("Could not get current directory");
    let cache_dir = dirs::cache_dir().expect("Could not find cache directory");
    let preview_dir = cache_dir.join("mapm").join("preview-all");
    if !preview_dir.is_dir() {
        fs::create_dir_all(&preview_dir).expect("Could not create directory");
    }
    fs::write(preview_dir.join("preview-all.tex"), &problems_tex).expect(&format!(
        "Could not write to `{}`",
        preview_dir.join("preview-all.tex").to_str().unwrap()
    ));
    env::set_current_dir(&preview_dir).expect("Could not set working directory");
    let latexmk = Command::new("latexmk")
        .args(["-pdf", "preview-all.tex"])
        .output()
        .expect("Failed to execute `latexmk`; make sure you have it installed");
    if latexmk.status.success() {
        println!(
            "`latexmk` on `{}` succeeded",
            preview_dir.join("preview-all.tex").to_str().unwrap()
        );
    } else {
        e_red_ln!(
            "`latexmk` on `{}` failed",
            preview_dir.join("preview-all.tex").to_str().unwrap()
        );
    }
    open::that(preview_dir.join("preview-all.pdf")).expect("Could not open `preview-all.pdf`");
    env::set_current_dir(&cwd).expect("Could not set working directory");
}
