//! Generates the templates that are used to interface with the `mapm` library
//!
//! Users are encouraged to modify these in the source code as they need; currently I can think of no better solution.

use colour::*;
use mapm::problem::{Solutions, Vars};
use std::collections::HashMap;

pub fn preview_template(problem_name: &str) -> String {
    ["template: preview\nvars: {}\nproblems:\n- ", problem_name].concat()
}

/// Gets a littany of `\usepackage`'s out of a Vec of &str's

fn package_list(packages: Vec<&str>) -> String {
    let mut package_list: String = String::new();
    for package in packages {
        package_list.push_str("\\usepackage{");
        package_list.push_str(package);
        package_list.push_str("}\n");
    }
    package_list
}

/// The impl of `problem_to_tex` is based on [Math Advance](mathadvance.org)'s usage patterns for mapm; you are encouraged to modify this as you see fit.

fn problem_to_tex(problem: (String, (Vars, Option<Solutions>))) -> String {
    let mut tex: String = String::new();
    let get_key = |key: &str| -> &str {
        match problem.1 .0.get(key) {
            Some(val) => val,
            None => {
                e_yellow_ln!("Could not find key `{}` in problem `{}`", key, &problem.0);
                ""
            }
        }
    };
    fn get_sol_key(
        key: &str,
        solution: &HashMap<String, String>,
        index: usize,
        name: &str,
    ) -> String {
        match solution.get(key) {
            Some(val) => String::from(val),
            None => {
                e_yellow_ln!(
                    "Could not find key `{}` in solution `{}` to problem `{}`",
                    key,
                    index + 1,
                    name
                );
                String::new()
            }
        }
    }
    tex.push_str("\\section{");
    tex.push_str(&problem.0);
    tex.push_str(" (");
    tex.push_str(get_key("author"));
    tex.push_str(")}\n");

    tex.push_str(get_key("problem"));
    tex.push_str("\n");

    let protected_keys: Vec<&str> = vec!["problem", "author", "answer"];
    let mut buckets: Vec<String> = Vec::new();
    let mut tags: Vec<(String, String)> = Vec::new();
    for (key, val) in problem.1 .0 {
        if !protected_keys.contains(&key.as_str()) {
            if val == "true" {
                buckets.push(key);
            } else {
                tags.push((key, val));
            }
        }
    }

    if buckets.len() > 0 {
        tex.push_str("\\subsection*{Buckets}\n");
        tex.push_str("\\begin{itemize}\n");

        for key in buckets {
            tex.push_str("\\item ");
            tex.push_str(&key);
            tex.push_str("\n");
        }

        tex.push_str("\\end{itemize}\n");
    }

    if tags.len() > 0 {
        tex.push_str("\\subsection*{Tags}\n");
        tex.push_str("\\begin{itemize}\n");

        for (key, val) in tags {
            tex.push_str("\\item ");
            tex.push_str(&key);
            tex.push_str(": ");
            tex.push_str(&val);
            tex.push_str("\n");
        }

        tex.push_str("\\end{itemize}\n");
    }

    match problem.1 .1 {
        Some(solutions) => {
            if solutions.len() > 0 {
                tex.push_str("\\subsection*{Solutions}\n");
                for (idx, solution) in solutions.iter().enumerate() {
                    tex.push_str("\\subsubsection*{Solution ");
                    tex.push_str(&(idx + 1).to_string());
                    tex.push_str(" (");
                    tex.push_str(&get_sol_key("author", &solution, idx, &problem.0));
                    tex.push_str(")}\n");
                    tex.push_str(&get_sol_key("text", &solution, idx, &problem.0));
                }
            }
        }
        None => {}
    }

    tex
}

/// If you want to modify this, DO NOT remove `\usepackage{mapm}`

pub fn preview_all_tex(problems: Vec<(String, (Vars, Option<Solutions>))>) -> String {
    let mut tex: String = [
        "\\documentclass{article}\n",
        &package_list(vec![
            "pgffor",
            "asymptote",
            "tikz",
            "amsmath",
            "amssymb",
            "float",
        ]),
        "\\newcommand{\\ansbold}[1]{\\mathbf{#1}}",
        "\\begin{document}\n",
    ]
    .concat();
    for problem in problems {
        tex.push_str(&problem_to_tex(problem));
        tex.push_str("\\newpage");
    }
    tex.push_str("\\end{document}");
    tex
}
