pub mod cli_display;
pub mod filter;
pub mod msg;
pub mod setup;
pub mod templates;
pub mod views;

use mapm::problem::FilterAction;
use mapm::problem::Views::*;
use mapm::problem::*;

use std::env;
use std::fs;
use std::path::*;
use std::process::Command;

use exitcode;
use quit;

use colour::*;

fn build_contest(contest_result: mapm::contest::ContestResult, template_dir: &Path) {
    match contest_result.contest {
        Some(contest) => {
            let contest_msgs = contest.compile(&template_dir);
            match contest_msgs.0 {
                Some(err) => {
                    msg::print_err(&err);
                }
                None => {}
            }
            for result in contest_msgs.1 {
                match result {
                    Ok(msg) => {
                        println!("{}", msg);
                    }
                    Err(err) => {
                        e_red_ln!("{}", err);
                    }
                }
            }
        }
        None => {
            match contest_result.contest_err {
                Some(err) => {
                    msg::print_err(&err);
                    quit::with_code(exitcode::DATAERR);
                }
                None => {}
            }
            match contest_result.problem_errs {
                Some(errs) => {
                    for problem_err in errs {
                        e_red_ln!("Errors for problem `{}`:", &problem_err.0);
                        for err in problem_err.1 {
                            msg::print_err(&err);
                        }
                    }
                }
                None => {}
            }
        }
    }
}

#[quit::main]
fn main() {
    let config = dirs::config_dir().unwrap();
    let profile: String;
    let profile_config_path = Path::new(&config).join("mapm").join("profile");
    if profile_config_path.is_file() {
        match fs::read_to_string(Path::new(&config).join("mapm").join("profile")) {
            Ok(profile_str) => {
                profile = profile_str;
            }
            Err(err) => {
                e_red_ln!("{}", err);
                quit::with_code(exitcode::IOERR);
            }
        }
    } else {
        setup::setup_prompt();
        return;
    }
    let problem_dir = Path::new(&config)
        .join("mapm")
        .join("problems")
        .join(&profile);
    if !problem_dir.is_dir() {
        fs::create_dir_all(&problem_dir).expect(
            &[
                "Could not create directory `",
                problem_dir.to_str().unwrap(),
                "`",
            ]
            .concat(),
        );
    }

    let template_dir = Path::new(&config).join("mapm").join("templates");

    if !template_dir.is_dir() {
        fs::create_dir_all(&template_dir).expect(&format!(
            "Could not create directory `{}`",
            &template_dir.to_str().unwrap()
        ));
    }

    let mut actions: Vec<(String, Vec<String>)> = Vec::new();
    for arg in env::args() {
        let arg_padded = [&arg, "  "].concat();
        if &arg_padded[0..2] == "--" {
            let len = actions.len();
            actions[len - 1].1.push(arg[2..].to_string());
        } else {
            actions.push((arg, Vec::new()));
        }
    }

    if actions[0].1.iter().any(|s| s == "help") {
        println!("{}", msg::HELP);
        quit::with_code(exitcode::OK);
    } else {
        if actions.len() == 1 {
            e_red_ln!("{}", msg::MISSING_SUBCMD);
            quit::with_code(exitcode::USAGE);
        } else {
            match actions[1].0.as_str() {
                "view" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_VIEW);
                        quit::with_code(exitcode::OK);
                    }
                    let views: Views;
                    if actions.len() > 3 {
                        e_red_ln!("{} (mapm view --help for help)", msg::TOO_MANY_ARGS);
                        quit::with_code(exitcode::USAGE);
                    } else if actions.len() == 3 {
                        match actions[2].0.as_str() {
                            "hide" => {
                                let mut views_vec: Vec<String> = Vec::new();
                                for view in &actions[2].1 {
                                    views_vec.push(String::from(view));
                                }
                                views = Hide(views_vec);
                            }
                            "show" => {
                                let mut views_vec: Vec<String> = Vec::new();
                                for view in &actions[2].1 {
                                    views_vec.push(String::from(view));
                                }
                                views = Show(views_vec);
                            }
                            _ => {
                                e_red_ln!("{}", msg::INVALID_SUBCMD);
                                quit::with_code(exitcode::USAGE);
                            }
                        }
                    } else {
                        views = Hide(Vec::new());
                    }
                    let problem_display = cli_display::problems_to_string(
                        views::problem_names_to_problems(&profile, &actions[1].1, views),
                    );
                    cli_display::display(&problem_display);
                }
                "find" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_FIND);
                        quit::with_code(exitcode::OK);
                    }
                    let views: Views;
                    if actions.len() > 3 {
                        e_red_ln!("{} (mapm find --help for help)", msg::TOO_MANY_ARGS);
                        quit::with_code(exitcode::USAGE);
                    } else if actions.len() == 3 {
                        views = views::parse_views(&actions[2]);
                    } else {
                        views = Hide(Vec::new());
                    }
                    let filters: Vec<FilterAction> = filter::parse_filters(&actions[1].1);
                    let problem_names = filter::filtered_names(&filters, &problem_dir);
                    let problem_display = cli_display::problems_to_string(
                        views::problem_names_to_problems(&profile, &problem_names, views),
                    );
                    cli_display::display(&problem_display);
                }
                "preview-all" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_FIND);
                        quit::with_code(exitcode::OK);
                    }
                    let views: Views;
                    if actions.len() > 3 {
                        e_red_ln!("{} (mapm find --help for help)", msg::TOO_MANY_ARGS);
                        quit::with_code(exitcode::USAGE);
                    } else if actions.len() == 3 {
                        views = views::parse_views(&actions[2]);
                    } else {
                        views = Hide(Vec::new());
                    }
                    let filters: Vec<FilterAction> = filter::parse_filters(&actions[1].1);
                    let problem_names = filter::filtered_names(&filters, &problem_dir);
                    let problems =
                        views::problem_names_to_problems(&profile, &problem_names, views);
                    let problems_tex = templates::preview_all_tex(problems);
                    let cwd = env::current_dir().expect("Could not get current directory");
                    let cache_dir = dirs::cache_dir().expect("Could not find cache directory");
                    let preview_dir = cache_dir.join("mapm").join("preview-all");
                    if !preview_dir.is_dir() {
                        fs::create_dir_all(&preview_dir).expect("Could not create directory");
                    }
                    fs::write(preview_dir.join("preview-all.tex"), &problems_tex).expect(&format!(
                        "Could not write to `{}`",
                        preview_dir.join("preview-all.tex").to_str().unwrap()
                    ));
                    env::set_current_dir(&preview_dir).expect("Could not set working directory");
                    let latexmk = Command::new("latexmk")
                        .args(["-pdf", "preview-all.tex"])
                        .output()
                        .expect("Failed to execute `latexmk`; make sure you have it installed");
                    if latexmk.status.success() {
                        println!(
                            "`latexmk` on `{}` succeeded",
                            preview_dir.join("preview-all.tex").to_str().unwrap()
                        );
                    } else {
                        e_red_ln!(
                            "`latexmk` on `{}` failed",
                            preview_dir.join("preview-all.tex").to_str().unwrap()
                        );
                    }
                    open::that(preview_dir.join("preview-all.pdf"))
                        .expect("Could not open `preview-all.pdf`");
                    env::set_current_dir(&cwd).expect("Could not set working directory");
                }
                "edit" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_EDIT);
                        quit::with_code(exitcode::OK);
                    }
                    if actions.len() > 3 {
                        e_red_ln!("{} (mapm edit --help for help)", msg::TOO_MANY_ARGS);
                        quit::with_code(exitcode::USAGE);
                    }
                    match actions.get(2) {
                        Some(action) => {
                            let problem_name = &action.0;
                            let problem_path_str = String::from(
                                problem_dir
                                    .join(&[problem_name.as_str(), ".yml"].concat())
                                    .to_str()
                                    .unwrap(),
                            );
                            let editor: String;
                            match env::var("EDITOR") {
                                Ok(var) => {
                                    editor = var;
                                }
                                Err(_) => {
                                    if cfg!(windows) {
                                        editor = String::from("notepad");
                                    } else {
                                        editor = String::from("nano");
                                    }
                                }
                            }
                            Command::new(&editor)
                                .arg(&problem_path_str)
                                .status()
                                .expect(&["Failed to start `", &editor, "`"].concat());
                        }
                        None => {
                            e_red_ln!("{}", msg::MISSING_EDIT_ARG);
                            quit::with_code(exitcode::USAGE);
                        }
                    }
                }
                "profile" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_PROFILE);
                        quit::with_code(exitcode::OK);
                    }
                    match actions.get(2) {
                        Some(action) => match action.0.as_str() {
                            "get" => {
                                if actions.len() > 3 {
                                    e_red_ln!(
                                        "{} (mapm profile --help for help)",
                                        msg::TOO_MANY_ARGS
                                    );
                                    quit::with_code(exitcode::USAGE);
                                }
                                println!("{}", &profile);
                            }
                            "set" => match actions.len() {
                                3 => {
                                    setup::setup_prompt();
                                }
                                4 => {
                                    setup::setup(&actions.get(3).unwrap().0);
                                }
                                _ => {
                                    e_red_ln!(
                                        "{} (mapm profile --help for help)",
                                        msg::TOO_MANY_ARGS
                                    );
                                    quit::with_code(exitcode::USAGE);
                                }
                            },
                            _ => {
                                e_red_ln!("{}", msg::INVALID_PROFILE_SUBCMD);
                                quit::with_code(exitcode::USAGE);
                            }
                        },
                        None => {
                            e_red_ln!("{}", msg::MISSING_PROFILE_ARG);
                            quit::with_code(exitcode::USAGE);
                        }
                    }
                }
                "build" => {
                    for action in &actions[2..] {
                        let contest_result = mapm::contest::fetch_contest(
                            &Path::new(&action.0),
                            &problem_dir,
                            &template_dir,
                        );
                        build_contest(contest_result, &template_dir);
                    }
                }
                "preview" => {
                    for action in &actions[2..] {
                        let problem_name = &action.0;
                        let contest_yml = &templates::preview_template(problem_name);
                        let contest_result = mapm::contest::parse_contest_yaml(
                            contest_yml,
                            &problem_dir,
                            &template_dir,
                        );
                        let cwd = env::current_dir().expect("Could not get current directory");
                        let cache_dir = dirs::cache_dir().expect("Could not find cache directory");
                        let preview_dir = cache_dir.join("mapm").join(problem_name);
                        if !preview_dir.is_dir() {
                            fs::create_dir_all(&preview_dir).expect("Could not create directory");
                        }
                        env::set_current_dir(&preview_dir)
                            .expect("Could not set working directory");
                        build_contest(contest_result, &template_dir);
                        for pdf in fs::read_dir(&preview_dir).expect("Could not read directory") {
                            let path = pdf.unwrap().path();
                            match &path.extension() {
                                Some(ext) => {
                                    if ext == &"pdf" {
                                        open::that(path).expect("Could not open PDF");
                                    }
                                }
                                None => {}
                            }
                        }
                        env::set_current_dir(&cwd).expect("Could not set working directory");
                    }
                }
                _ => {
                    e_red_ln!("{}", msg::INVALID_SUBCMD);
                    quit::with_code(exitcode::USAGE);
                }
            }
        }
    }
}
