pub mod display;
pub mod msg;
pub mod setup;

use mapm::problem::Filter;
use mapm::problem::FilterAction;
use mapm::problem::Views::*;
use mapm::problem::*;
use mapm::result::MapmErr::*;
use mapm::result::MapmResult::*;

use std::env;
use std::fs;
use std::path::*;
use std::process::Command;

use exitcode;
use quit;

use ansi_term::Colour::*;
use colour::*;

fn build_contest(contest_result: mapm::contest::ContestResult, template_dir: &Path) {
    match contest_result.contest {
        Some(contest) => {
            let contest_msgs = contest.compile(&template_dir);
            match contest_msgs.0 {
                Some(err) => {
                    msg::print_err(&err);
                }
                None => {}
            }
            for result in contest_msgs.1 {
                match result {
                    Ok(msg) => {
                        println!("{}", msg);
                    }
                    Err(err) => {
                        e_red_ln!("{}", err);
                    }
                }
            }
        }
        None => {
            match contest_result.contest_err {
                Some(err) => {
                    msg::print_err(&err);
                    quit::with_code(exitcode::DATAERR);
                }
                None => {}
            }
            match contest_result.problem_errs {
                Some(errs) => {
                    for problem_err in errs {
                        e_red_ln!("Errors for problem `{}`:", &problem_err.0);
                        for err in problem_err.1 {
                            msg::print_err(&err);
                        }
                    }
                }
                None => {}
            }
        }
    }
}

fn get_problems_string(profile: &str, problem_names: &Vec<String>, views: Views) -> String {
    let home = dirs::home_dir().unwrap();
    let mut problem_display = String::new();
    let mut problem_index = 0;
    for problem_name in problem_names {
        problem_index += 1;
        if problem_index > 1 {
            problem_display.push_str("\n\n");
        }
        let problem_result = fetch_problem(
            problem_name,
            &Path::new(&home)
                .join(".mapm")
                .join("problems")
                .join(profile),
        );
        match problem_result {
            Success(problem) => {
                let filtered_problem = problem.filter_keys(views.clone());
                problem_display.push_str(
                    &Green
                        .bold()
                        .paint(["-- ", problem_name, " --"].concat())
                        .to_string(),
                );
                for (key, val) in &filtered_problem.0 {
                    problem_display.push_str("\n");
                    problem_display.push_str(&Blue.paint(key).to_string());
                    problem_display.push_str(": ");
                    let mut idx = 0;
                    for string in val.split("\n") {
                        idx += 1;
                        if idx == 1 {
                            problem_display.push_str(&string);
                        } else {
                            problem_display.push_str("\n  ");
                            problem_display.push_str(&str::repeat(" ", key.chars().count()));
                            problem_display.push_str(&string);
                        }
                    }
                }
                match &filtered_problem.1 {
                    Some(solutions) => {
                        problem_display.push_str("\n");
                        problem_display.push_str(&Blue.bold().paint("solutions").to_string());
                        problem_display.push_str(": ");
                        if solutions.len() == 0 {
                            problem_display.push_str(&Red.bold().paint("empty").to_string());
                        } else {
                            let mut index: u32 = 0;
                            for solution in solutions {
                                index += 1;
                                problem_display.push_str("\n  ");
                                problem_display.push_str(
                                    &Red.paint([&index.to_string(), "."].concat()).to_string(),
                                );
                                let mut key_index = 0;
                                for (key, val) in solution {
                                    key_index += 1;
                                    if key_index == 1 {
                                        problem_display.push_str(" ");
                                    } else {
                                        problem_display.push_str("\n");
                                        problem_display.push_str(&str::repeat(" ", 5));
                                    }
                                    problem_display.push_str(&Blue.paint(key).to_string());
                                    problem_display.push_str(": ");
                                    let mut idx = 0;
                                    for string in val.split("\n") {
                                        idx += 1;
                                        if idx == 1 {
                                            problem_display.push_str(&string);
                                        } else {
                                            problem_display.push_str("\n");
                                            problem_display.push_str(&str::repeat(
                                                " ",
                                                index.to_string().len() + key.chars().count() + 6,
                                            ));
                                            problem_display.push_str(&string);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    None => {
                        problem_display.push_str("\n");
                    }
                }
            }
            Fail(err) => match err {
                ProblemErr(msg) => {
                    e_yellow_ln!("{}", msg);
                }
                _ => {
                    e_magenta_ln!("{}", msg::VIEW_NOT_PROBLEMERR);
                    quit::with_code(exitcode::SOFTWARE);
                }
            },
        }
    }
    return problem_display;
}

#[quit::main]
fn main() {
    let home = dirs::home_dir().unwrap();
    let profile: String;
    let profile_config_path = Path::new(&home).join(".mapm").join("profile");
    if profile_config_path.is_file() {
        match fs::read_to_string(Path::new(&home).join(".mapm").join("profile")) {
            Ok(profile_str) => {
                profile = profile_str;
            }
            Err(err) => {
                e_red_ln!("{}", err);
                quit::with_code(exitcode::IOERR);
            }
        }
    } else {
        setup::setup();
        return;
    }
    let problem_dir = Path::new(&home)
        .join(".mapm")
        .join("problems")
        .join(&profile);
    if !problem_dir.is_dir() {
        fs::create_dir_all(&problem_dir).expect(
            &[
                "Could not create directory `",
                problem_dir.to_str().unwrap(),
                "`",
            ]
            .concat(),
        );
    }

    let template_dir = Path::new(&home).join(".mapm").join("templates");

    if !template_dir.is_dir() {
        fs::create_dir_all(&template_dir).expect("Could not create directory `~/.mapm/templates`");
    }

    let mut actions: Vec<(String, Vec<String>)> = Vec::new();
    for arg in env::args() {
        let arg_padded = [&arg, "  "].concat();
        if &arg_padded[0..2] == "--" {
            let len = actions.len();
            actions[len - 1].1.push(arg[2..].to_string());
        } else {
            actions.push((arg, Vec::new()));
        }
    }

    if actions[0].1.iter().any(|s| s == "help") {
        println!("{}", msg::HELP);
        quit::with_code(exitcode::OK);
    } else {
        if actions.len() == 1 {
            e_red_ln!("{}", msg::MISSING_SUBCMD);
            quit::with_code(exitcode::USAGE);
        } else {
            match actions[1].0.as_str() {
                "view" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_VIEW);
                        quit::with_code(exitcode::OK);
                    }
                    let views: Views;
                    if actions.len() > 3 {
                        e_red_ln!("{} (mapm view --help for help)", msg::TOO_MANY_ARGS);
                        quit::with_code(exitcode::USAGE);
                    } else if actions.len() == 3 {
                        match actions[2].0.as_str() {
                            "hide" => {
                                let mut views_vec: Vec<String> = Vec::new();
                                for view in &actions[2].1 {
                                    views_vec.push(String::from(view));
                                }
                                views = Hide(views_vec);
                            }
                            "show" => {
                                let mut views_vec: Vec<String> = Vec::new();
                                for view in &actions[2].1 {
                                    views_vec.push(String::from(view));
                                }
                                views = Show(views_vec);
                            }
                            _ => {
                                e_red_ln!("{}", msg::INVALID_SUBCMD);
                                quit::with_code(exitcode::USAGE);
                            }
                        }
                    } else {
                        views = Hide(Vec::new());
                    }
                    let problem_display = get_problems_string(&profile, &actions[1].1, views);
                    display::display(&problem_display);
                }
                "find" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_FIND);
                        quit::with_code(exitcode::OK);
                    }
                    let views: Views;
                    if actions.len() > 3 {
                        e_red_ln!("{} (mapm view --help for help)", msg::TOO_MANY_ARGS);
                        quit::with_code(exitcode::USAGE);
                    } else if actions.len() == 3 {
                        match actions[2].0.as_str() {
                            "hide" => {
                                let mut views_vec: Vec<String> = Vec::new();
                                for view in &actions[2].1 {
                                    views_vec.push(String::from(view));
                                }
                                views = Hide(views_vec);
                            }
                            "show" => {
                                let mut views_vec: Vec<String> = Vec::new();
                                for view in &actions[2].1 {
                                    views_vec.push(String::from(view));
                                }
                                views = Show(views_vec);
                            }
                            _ => {
                                e_red_ln!("{}", msg::INVALID_SUBCMD);
                                quit::with_code(exitcode::USAGE);
                            }
                        }
                    } else {
                        views = Hide(Vec::new());
                    }
                    let mut filters: Vec<FilterAction> = Vec::new();

                    fn parse_filter(arg: &str) -> Filter {
                        match arg.find("<=") {
                            Some(idx) => match arg[idx + 2..].parse::<u32>() {
                                Ok(val) => {
                                    return Filter::Le {
                                        key: String::from(&arg[0..idx]),
                                        val,
                                    };
                                }
                                Err(_) => {
                                    e_red_ln!("Filter key `{}` was passed with non-integer value `{}`, despite non-equality comparison operator `<=`", &arg[0..idx], &arg[idx+2..]);
                                    quit::with_code(exitcode::USAGE);
                                }
                            },
                            None => match arg.find(">=") {
                                Some(idx) => match arg[idx + 2..].parse::<u32>() {
                                    Ok(val) => {
                                        return Filter::Ge {
                                            key: String::from(&arg[0..idx]),
                                            val,
                                        };
                                    }
                                    Err(_) => {
                                        e_red_ln!("Filter key `{}` was passed with non-integer value `{}`, despite non-equality comparison operator `>=`", &arg[0..idx], &arg[idx+2..]);
                                        quit::with_code(exitcode::USAGE);
                                    }
                                },

                                None => match arg.find("=") {
                                    Some(idx) => {
                                        return Filter::Eq {
                                            key: String::from(&arg[0..idx]),
                                            val: String::from(&arg[idx + 1..]),
                                        }
                                    }
                                    None => match arg.find("<") {
                                        Some(idx) => match arg[idx + 1..].parse::<u32>() {
                                            Ok(val) => {
                                                return Filter::Le {
                                                    key: String::from(&arg[0..idx]),
                                                    val,
                                                };
                                            }
                                            Err(_) => {
                                                e_red_ln!("Filter key `{}` was passed with non-integer value `{}`, despite non-equality comparison operator `<`", &arg[0..idx], &arg[idx+1..]);
                                                quit::with_code(exitcode::USAGE);
                                            }
                                        },
                                        None => match arg.find(">") {
                                            Some(idx) => match arg[idx + 1..].parse::<u32>() {
                                                Ok(val) => {
                                                    return Filter::Le {
                                                        key: String::from(&arg[0..idx]),
                                                        val,
                                                    };
                                                }
                                                Err(_) => {
                                                    e_red_ln!("Filter key `{}` was passed with non-integer value `{}`, despite non-equality comparison operator `>`", &arg[0..idx], &arg[idx+1..]);
                                                    quit::with_code(exitcode::USAGE);
                                                }
                                            },
                                            None => {
                                                return Filter::Exists {
                                                    key: String::from(arg),
                                                }
                                            }
                                        },
                                    },
                                },
                            },
                        }
                    }

                    for arg in &actions[1].1 {
                        if &arg[0..1] == "!" {
                            filters.push(FilterAction::Negative(parse_filter(&arg[1..])));
                        } else {
                            filters.push(FilterAction::Positive(parse_filter(arg)));
                        }
                    }
                    let mut problem_names: Vec<String> = Vec::new();
                    for file in fs::read_dir(problem_dir).unwrap() {
                        let problem_yaml =
                            fs::read_to_string(file.as_ref().unwrap().path()).unwrap();
                        let problem_res = mapm::problem::parse_problem_yaml(&problem_yaml);
                        match problem_res {
                            Success(problem) => match problem.filter(filters.clone()) {
                                Some(_) => {
                                    problem_names.push(String::from(
                                        file.as_ref()
                                            .unwrap()
                                            .path()
                                            .file_stem()
                                            .unwrap()
                                            .to_str()
                                            .unwrap(),
                                    ));
                                }
                                None => {}
                            },
                            Fail(err) => match err {
                                ProblemErr(msg) => {
                                    e_yellow_ln!("{}", msg);
                                }
                                _ => {
                                    e_magenta_ln!("{}", msg::FIND_NOT_PROBLEMERR);
                                    quit::with_code(exitcode::SOFTWARE);
                                }
                            },
                        }
                    }
                    let problem_display = get_problems_string(&profile, &problem_names, views);
                    display::display(&problem_display);
                }
                "edit" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_EDIT);
                        quit::with_code(exitcode::OK);
                    }
                    if actions.len() > 3 {
                        e_red_ln!("{} (mapm edit --help for help)", msg::TOO_MANY_ARGS);
                        quit::with_code(exitcode::USAGE);
                    }
                    match actions.get(2) {
                        Some(action) => {
                            let problem_name = &action.0;
                            let problem_path_str = String::from(
                                problem_dir
                                    .join(&[problem_name.as_str(), ".yml"].concat())
                                    .to_str()
                                    .unwrap(),
                            );
                            let editor: String;
                            match env::var("EDITOR") {
                                Ok(var) => {
                                    editor = var;
                                }
                                Err(_) => {
                                    if cfg!(windows) {
                                        editor = String::from("notepad");
                                    } else {
                                        editor = String::from("nano");
                                    }
                                }
                            }
                            Command::new(&editor)
                                .arg(&problem_path_str)
                                .status()
                                .expect(&["Failed to start `", &editor, "`"].concat());
                        }
                        None => {
                            e_red_ln!("{}", msg::MISSING_EDIT_ARG);
                            quit::with_code(exitcode::USAGE);
                        }
                    }
                }
                "profile" => {
                    if actions[1].1.iter().any(|s| s == "help") {
                        println!("{}", msg::HELP_PROFILE);
                        quit::with_code(exitcode::OK);
                    }
                    match actions.get(2) {
                        Some(action) => match action.0.as_str() {
                            "get" => {
                                if actions.len() > 3 {
                                    e_red_ln!(
                                        "{} (mapm profile --help for help)",
                                        msg::TOO_MANY_ARGS
                                    );
                                    quit::with_code(exitcode::USAGE);
                                }
                                println!("{}", &profile);
                            }
                            "set" => {
                                if actions.len() > 3 {
                                    e_red_ln!(
                                        "{} (mapm profile --help for help)",
                                        msg::TOO_MANY_ARGS
                                    );
                                    quit::with_code(exitcode::USAGE);
                                }
                                setup::setup();
                            }
                            _ => {
                                e_red_ln!("{}", msg::INVALID_PROFILE_SUBCMD);
                                quit::with_code(exitcode::USAGE);
                            }
                        },
                        None => {
                            e_red_ln!("{}", msg::MISSING_PROFILE_ARG);
                            quit::with_code(exitcode::USAGE);
                        }
                    }
                }
                "build" => {
                    for action in &actions[2..] {
                        let contest_result = mapm::contest::fetch_contest(
                            &Path::new(&action.0),
                            &problem_dir,
                            &template_dir,
                        );
                        build_contest(contest_result, &template_dir);
                    }
                }
                "preview" => {
                    for action in &actions[2..] {
                        let problem_name = &action.0;
                        let contest_yml = &[
                            "template: preview
vars: {}
problems:
  - ",
                            problem_name,
                        ]
                        .concat();
                        let contest_result = mapm::contest::parse_contest_yaml(
                            contest_yml,
                            &problem_dir,
                            &template_dir,
                        );
                        let cwd = env::current_dir().expect("Could not get current directory");
                        let cache_dir = dirs::cache_dir().expect("Could not find cache directory");
                        let preview_dir = cache_dir.join("mapm").join(problem_name);
                        if !preview_dir.is_dir() {
                            fs::create_dir_all(&preview_dir).expect("Could not create directory");
                        }
                        env::set_current_dir(&preview_dir)
                            .expect("Could not set working directory");
                        build_contest(contest_result, &template_dir);
                        for pdf in fs::read_dir(&preview_dir).expect("Could not read directory") {
                            let path = pdf.unwrap().path();
                            match &path.extension() {
                                Some(ext) => {
                                    if ext == &"pdf" {
                                        open::that(path).expect("Could not open PDF");
                                    }
                                }
                                None => {}
                            }
                        }
                        env::set_current_dir(&cwd).expect("Could not set working directory");
                    }
                }
                _ => {
                    e_red_ln!("{}", msg::INVALID_SUBCMD);
                    quit::with_code(exitcode::USAGE);
                }
            }
        }
    }
}
