//! Author endpoint handler.
//!
//! <https://api.mangadex.org/docs.html#tag/Author>

mod create;
mod delete;
mod get;
pub(crate) mod list;
mod update;

use crate::v5::author::create::CreateAuthorBuilder;
use crate::v5::author::delete::DeleteAuthorBuilder;
use crate::v5::author::get::GetAuthorBuilder;
use crate::v5::author::list::ListAuthorBuilder;
use crate::v5::author::update::UpdateAuthorBuilder;
use crate::HttpClientRef;

/// Author endpoint handler builder.
#[derive(Debug)]
pub struct AuthorBuilder {
    http_client: HttpClientRef,
}

impl AuthorBuilder {
    #[doc(hidden)]
    pub(crate) fn new(http_client: HttpClientRef) -> Self {
        Self { http_client }
    }

    /// Search for authors.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-author>
    pub fn list(&self) -> ListAuthorBuilder {
        ListAuthorBuilder::default().http_client(self.http_client.clone())
    }

    /// Search for authors.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-author>
    ///
    /// This is an alias for the [`Self::list()`] function.
    pub fn search(&self) -> ListAuthorBuilder {
        self.list()
    }

    /// View a single author.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-author-id>
    pub fn get(&self) -> GetAuthorBuilder {
        GetAuthorBuilder::default().http_client(self.http_client.clone())
    }

    /// View a single author.
    ///
    /// <https://api.mangadex.org/docs.html#operation/get-author-id>
    ///
    /// This is an alias for [`Self::get()`] to maintain backwards-compatibility.
    pub fn view(&self) -> GetAuthorBuilder {
        self.get()
    }

    /// Create an author.
    ///
    /// <https://api.mangadex.org/docs.html#post-author>
    pub fn create(&self) -> CreateAuthorBuilder {
        CreateAuthorBuilder::default().http_client(self.http_client.clone())
    }

    /// Update an author.
    ///
    /// <https://api.mangadex.org/docs.html#operation/put-author-id>
    pub fn update(&self) -> UpdateAuthorBuilder {
        UpdateAuthorBuilder::default().http_client(self.http_client.clone())
    }

    /// Delete an author.
    ///
    /// <https://api.mangadex.org/docs.html#operation/delete-author-id>
    pub fn delete(&self) -> DeleteAuthorBuilder {
        DeleteAuthorBuilder::default().http_client(self.http_client.clone())
    }
}
