//! Builder for the chapter view endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-chapter-id>
//!
//! # Examples
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let chapter_id = Uuid::new_v4();
//! let chapter_res = client
//!     .chapter()
//!     .get()
//!     .chapter_id(&chapter_id)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("chapter: {:?}", chapter_res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::types::RelationshipType;
use crate::v5::schema::ChapterResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned")]
pub struct GetChapter<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[serde(skip)]
    pub chapter_id: &'a Uuid,

    #[builder(setter(each = "add_relationship"), default)]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET ("/chapter/{}", chapter_id),
    #[query] GetChapter<'_>,
    #[flatten_result] ChapterResponse
}

#[cfg(test)]
mod tests {
    #[cfg(feature = "chrono")]
    use chrono::{DateTime, Utc};
    #[cfg(not(feature = "time"))]
    use fake::faker::chrono::en::DateTime;
    use fake::faker::internet::en::Password;
    use fake::faker::name::en::Name;
    use fake::Fake;
    use serde_json::json;
    #[cfg(feature = "time")]
    use time::OffsetDateTime;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path_regex};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::types::{Language, MangaDexDateTime, ResponseType};
    use crate::{Error, HttpClient, MangaDexClient};

    #[tokio::test]
    async fn get_chapter_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let chapter_id = Uuid::new_v4();
        let uploader_id = Uuid::new_v4();
        let chapter_title: String = Name().fake();
        let hash: String = Password(16..24).fake();

        let datetime = {
            #[cfg(not(any(feature = "chrono", feature = "time")))]
            let datetime: String = DateTime().fake();
            #[cfg(feature = "chrono")]
            let datetime: DateTime<Utc> = DateTime().fake();
            #[cfg(feature = "time")]
            let datetime = OffsetDateTime::now_utc();

            MangaDexDateTime::new(&datetime)
        };

        let response_body = json!({
            "result": "ok",
            "response": "entity",
            "data": {
                "id": chapter_id,
                "type": "chapter",
                "attributes": {
                    "title": chapter_title,
                    "volume": "1",
                    "chapter": "1.5",
                    "translatedLanguage": "en",
                    "hash": hash,
                    "data": [
                        "1.jpg"
                    ],
                    "dataSaver": [
                        "1.jpg"
                    ],
                    "uploader": uploader_id,
                    "version": 1,
                    "createdAt": datetime.to_string(),
                    "updatedAt": datetime.to_string(),
                    "publishAt": datetime.to_string(),
                },
                "relationships": []
            }
        });

        Mock::given(method("GET"))
            .and(path_regex(r"/chapter/[0-9a-fA-F-]+"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .chapter()
            .get()
            .chapter_id(&chapter_id)
            .build()?
            .send()
            .await?;

        assert_eq!(res.response, ResponseType::Entity);
        assert_eq!(res.data.id, chapter_id);
        assert_eq!(res.data.attributes.title, chapter_title);
        assert_eq!(res.data.attributes.volume, Some("1".to_string()));
        assert_eq!(res.data.attributes.chapter, Some("1.5".to_string()));
        assert_eq!(res.data.attributes.translated_language, Language::English);
        assert_eq!(res.data.attributes.hash, hash);
        assert_eq!(res.data.attributes.data, vec!["1.jpg"]);
        assert_eq!(res.data.attributes.data_saver, vec!["1.jpg"]);
        assert_eq!(res.data.attributes.uploader, Some(uploader_id));
        assert_eq!(res.data.attributes.version, 1);
        assert_eq!(
            res.data.attributes.created_at.to_string(),
            datetime.to_string()
        );
        assert_eq!(
            res.data.attributes.updated_at.as_ref().unwrap().to_string(),
            datetime.to_string()
        );
        assert_eq!(
            res.data.attributes.publish_at.to_string(),
            datetime.to_string()
        );

        Ok(())
    }

    #[tokio::test]
    async fn get_chapter_handles_404() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let chapter_id = Uuid::new_v4();
        let error_id = Uuid::new_v4();

        let response_body = json!({
            "result": "error",
            "errors": [{
                "id": error_id.to_string(),
                "status": 404,
                "title": "Not found",
                "detail": "Chapter could not be found"
            }]
        });

        Mock::given(method("GET"))
            .and(path_regex(r"/chapter/[0-9a-fA-F-]+"))
            .respond_with(ResponseTemplate::new(404).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .chapter()
            .get()
            .chapter_id(&chapter_id)
            .build()?
            .send()
            .await
            .expect_err("expected error");

        if let Error::Api(errors) = res {
            assert_eq!(errors.errors.len(), 1);

            assert_eq!(errors.errors[0].id, error_id);
            assert_eq!(errors.errors[0].status, 404);
            assert_eq!(errors.errors[0].title, Some("Not found".to_string()));
            assert_eq!(
                errors.errors[0].detail,
                Some("Chapter could not be found".to_string())
            );
        }

        Ok(())
    }
}
