//! Builder for the user manga feed endpoint to get a list of new chapters for a user's manga.
//!
//! <https://api.mangadex.org/docs.html#operation/get-user-follows-manga-feed>
//!
//! # Examples
//!
//! ```rust
//! use mangadex_api::v5::{MangaDexClient, Password, Username};
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let _login_res = client
//!     .auth()
//!     .login()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter23")?)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let res = client
//!     .user()
//!     .followed_manga_feed()
//!     .limit(1u32)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("Manga feed: {:?}", res);
//! # Ok(())
//! # }
//! ```

use chrono::{DateTime, Utc};
use derive_builder::Builder;
use serde::Serialize;

use crate::common::serialize_mangadex_datetime;
use crate::types::{
    ContentRating, IncludeFutureUpdates, Language, MangaFeedSortOrder, RelationshipType,
};
use crate::v5::schema::ChapterListResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder, Default)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned", default)]
pub struct GetFollowedMangaFeed {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub limit: Option<u32>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub offset: Option<u32>,
    #[builder(setter(each = "add_translated_language"))]
    #[serde(skip_serializing_if = "Vec::is_empty")]
    pub translated_language: Vec<Language>,
    #[builder(setter(each = "add_original_language"))]
    #[serde(skip_serializing_if = "Vec::is_empty")]
    pub original_language: Vec<Language>,
    #[builder(setter(each = "exclude_original_language"))]
    #[serde(skip_serializing_if = "Vec::is_empty")]
    pub excluded_original_language: Vec<Language>,
    #[builder(setter(each = "add_content_rating"))]
    pub content_rating: Vec<ContentRating>,
    /// Flag to include future chapter updates in the results.
    ///
    /// Default: `IncludeFutureUpdates::Include` (1)
    pub include_future_updates: Option<IncludeFutureUpdates>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[serde(serialize_with = "serialize_mangadex_datetime")]
    pub created_at_since: Option<DateTime<Utc>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[serde(serialize_with = "serialize_mangadex_datetime")]
    pub updated_at_since: Option<DateTime<Utc>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[serde(serialize_with = "serialize_mangadex_datetime")]
    pub publish_at_since: Option<DateTime<Utc>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub order: Option<MangaFeedSortOrder>,
    #[builder(setter(each = "add_relationship"))]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET "/user/follows/manga/feed",
    #[query auth] GetFollowedMangaFeed,
    #[flatten_result] ChapterListResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::faker::internet::en::Password;
    use fake::faker::name::en::Name;
    use fake::Fake;
    use regex::Regex;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{header, method, path_regex};
    use wiremock::{Mock, MockServer, Request, ResponseTemplate};

    use crate::v5::AuthTokens;
    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn get_followed_manga_feed_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let chapter_id = Uuid::new_v4();
        let uploader_id = Uuid::new_v4();
        let chapter_title: String = Name().fake();
        let hash: String = Password(16..24).fake();
        let date_time: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "result": "ok",
            "response": "collection",
            "data": [
                {
                    "id": chapter_id,
                    "type": "chapter",
                    "attributes": {
                        "title": chapter_title,
                        "volume": "1",
                        "chapter": "1.5",
                        "translatedLanguage": "en",
                        "hash": hash,
                        "data": [
                            "1.jpg"
                        ],
                        "dataSaver": [
                            "1.jpg"
                        ],
                        "uploader": uploader_id,
                        "version": 1,
                        "createdAt": date_time,
                        "updatedAt": date_time,
                        "publishAt": date_time
                    },
                    "relationships": []
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        });

        Mock::given(method("GET"))
            .and(path_regex(r"/user/follows/manga/feed"))
            .and(header("Authorization", "Bearer sessiontoken"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let _ = mangadex_client
            .user()
            .followed_manga_feed()
            .limit(1u32)
            .build()?
            .send()
            .await?;

        Ok(())
    }

    async fn followed_manga_feed_datetime_setup(client: &MangaDexClient) {
        let date_time: DateTime<Utc> = DateTime().fake();
        let _ = client
            .user()
            .followed_manga_feed()
            .created_at_since(date_time)
            .updated_at_since(date_time)
            .publish_at_since(date_time)
            .build()
            .unwrap()
            .send()
            .await;
    }

    test_serialize_mangadex_datetime!(
        #[query_string auth]
        ["createdAtSince", "updatedAtSince", "publishAtSince"],
        followed_manga_feed_datetime_setup
    );
}
