//! Builder for the manga list endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-search-manga>
//!
//! # Examples
//!
//! ```rust
//! use mangadex_api::types::MangaStatus;
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let manga_res = client
//!     .manga()
//!     .list()
//!     .title("full metal")
//!     .add_status(MangaStatus::Completed)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("manga: {:?}", manga_res);
//! # Ok(())
//! # }
//! ```

use chrono::{DateTime, Utc};
use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::common::serialize_mangadex_datetime;
use crate::types::{
    ContentRating, Demographic, Language, MangaSortOrder, MangaStatus, RelationshipType, Tag,
    TagSearchMode,
};
use crate::v5::schema::MangaListResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder, Default)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), default, pattern = "owned")]
pub struct ListManga {
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    pub limit: Option<u32>,
    pub offset: Option<u32>,
    pub title: Option<String>,
    #[builder(setter(each = "add_author"))]
    pub authors: Vec<Uuid>,
    #[builder(setter(each = "add_artist"))]
    pub artists: Vec<Uuid>,
    pub year: Option<u16>,
    #[builder(setter(each = "include_tag"))]
    pub included_tags: Vec<Tag>,
    pub included_tags_mode: Option<TagSearchMode>,
    #[builder(setter(each = "exclude_tag"))]
    pub excluded_tags: Vec<Tag>,
    pub excluded_tags_mode: Option<TagSearchMode>,
    #[builder(setter(each = "add_status"))]
    pub status: Vec<MangaStatus>,
    /// Languages the manga results are originally published in.
    #[builder(setter(each = "add_original_language"))]
    pub original_language: Vec<Language>,
    /// A list of original languages to exclude.
    #[builder(setter(each = "exclude_original_language"))]
    pub excluded_original_language: Vec<Language>,
    /// A list of languages that the manga is translated into.
    #[builder(setter(each = "add_available_translated_language"))]
    pub available_translated_language: Vec<Language>,
    #[builder(setter(each = "add_publication_demographic"))]
    pub publication_demographic: Vec<Demographic>,
    #[builder(setter(each = "add_manga_id"))]
    #[serde(rename = "ids")]
    pub manga_ids: Vec<Uuid>,
    #[builder(setter(each = "add_content_rating"))]
    pub content_rating: Vec<ContentRating>,
    #[serde(serialize_with = "serialize_mangadex_datetime")]
    pub created_at_since: Option<DateTime<Utc>>,
    #[serde(serialize_with = "serialize_mangadex_datetime")]
    pub updated_at_since: Option<DateTime<Utc>>,
    pub order: Option<MangaSortOrder>,
    #[builder(setter(each = "add_relationship"))]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET "/manga",
    #[query] ListManga,
    #[flatten_result] MangaListResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::Fake;
    use regex::Regex;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path};
    use wiremock::{Mock, MockServer, Request, ResponseTemplate};

    use crate::types::{ContentRating, Demographic, Language, MangaStatus, ResponseType};
    use crate::{Error, HttpClient, MangaDexClient};

    #[tokio::test]
    async fn list_manga_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let manga_id = Uuid::new_v4();
        let manga_title = "Test Manga".to_string();
        let created_at: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "result": "ok",
            "response": "collection",
            "data": [
                {
                    "id": manga_id,
                    "type": "manga",
                    "attributes": {
                        "title": {
                            "en": manga_title
                        },
                        "altTitles": [],
                        "description": {},
                        "isLocked": false,
                        "links": null,
                        "originalLanguage": "ja",
                        "lastVolume": null,
                        "lastChapter": null,
                        "publicationDemographic": "shoujo",
                        "status": "ongoing",
                        "year": null,
                        "contentRating": "safe",
                        "tags": [],
                        "state": "published",
                        "createdAt": created_at,
                        "updatedAt": created_at,
                        "version": 1
                    },
                    "relationships": []
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        });

        Mock::given(method("GET"))
            .and(path("/manga"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .manga()
            .search()
            .limit(1u32)
            .build()?
            .send()
            .await?;

        assert_eq!(res.response, ResponseType::Collection);
        let manga = &res.data[0];
        assert_eq!(manga.id, manga_id);
        assert_eq!(
            manga.attributes.title.get(&Language::English).unwrap(),
            &manga_title
        );
        assert!(manga.attributes.alt_titles.is_empty());
        assert!(manga.attributes.description.is_empty());
        assert!(!manga.attributes.is_locked);
        assert_eq!(manga.attributes.links, None);
        assert_eq!(manga.attributes.original_language, Language::Japanese);
        assert_eq!(manga.attributes.last_volume, None);
        assert_eq!(manga.attributes.last_chapter, None);
        assert_eq!(
            manga.attributes.publication_demographic.unwrap(),
            Demographic::Shoujo
        );
        assert_eq!(manga.attributes.status, MangaStatus::Ongoing);
        assert_eq!(manga.attributes.year, None);
        assert_eq!(
            manga.attributes.content_rating.unwrap(),
            ContentRating::Safe
        );
        assert!(manga.attributes.tags.is_empty());
        assert_eq!(manga.attributes.created_at, created_at);
        assert_eq!(manga.attributes.updated_at.unwrap(), created_at);
        assert_eq!(manga.attributes.version, 1);

        Ok(())
    }

    #[tokio::test]
    async fn list_manga_handles_400() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let error_id = Uuid::new_v4();

        let response_body = json!({
            "result": "error",
            "errors": [{
                "id": error_id.to_string(),
                "status": 400,
                "title": "Invalid limit",
                "detail": "Limit must be between 1 and 100"
            }]
        });

        Mock::given(method("GET"))
            .and(path("/manga"))
            .respond_with(ResponseTemplate::new(400).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .manga()
            .search()
            .limit(0u32)
            .build()?
            .send()
            .await
            .expect_err("expected error");

        if let Error::Api(errors) = res {
            assert_eq!(errors.errors.len(), 1);

            assert_eq!(errors.errors[0].id, error_id);
            assert_eq!(errors.errors[0].status, 400);
            assert_eq!(errors.errors[0].title, Some("Invalid limit".to_string()));
            assert_eq!(
                errors.errors[0].detail,
                Some("Limit must be between 1 and 100".to_string())
            );
        }

        Ok(())
    }

    async fn list_manga_datetime_setup(client: &MangaDexClient) {
        let date_time: DateTime<Utc> = DateTime().fake();
        let _ = client
            .manga()
            .search()
            .created_at_since(date_time)
            .updated_at_since(date_time)
            .build()
            .unwrap()
            .send()
            .await;
    }

    test_serialize_mangadex_datetime!(
        #[query_string]
        ["createdAtSince", "updatedAtSince"],
        list_manga_datetime_setup
    );
}
