use crate::v5::schema::MangaDexErrorResponse;

pub type Result<T, E = Error> = std::result::Result<T, E>;

#[derive(Debug, thiserror::Error)]
pub enum Error {
    /// Error when parsing a URL.
    ///
    /// This should not happen.
    #[error("error parsing the URL")]
    ParseUrlError(#[from] url::ParseError),

    #[error("failed to send a request to MangaDex: {0}")]
    RequestError(#[from] reqwest::Error),

    /// Error when building the request.
    #[error("failed to build the request: {0}")]
    BuilderError(#[from] derive_builder::UninitializedFieldError),

    #[error("missing auth tokens; please log in to MangaDex")]
    MissingTokens,

    #[error("not a valid username: {0}")]
    UsernameError(String),

    #[error("not a valid password: {0}")]
    PasswordError(String),

    #[error("an error occurred while pinging the MangaDex server")]
    PingError,

    /// Errors returned from the MangaDex API request.
    #[error("an error occurred with the MangaDex API request: {0}")]
    Api(#[from] MangaDexErrorResponse),

    /// Error while building the request struct.
    #[error("failed to build a request: {0}")]
    RequestBuilderError(String),

    /// Error while parsing the type.
    #[error("an error occurred while parsing the type: {0}")]
    ParseError(String),

    #[error(transparent)]
    UnexpectedError(#[from] anyhow::Error),
}
