//! Builder for the cover view endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-cover-id>
//!
//! # Examples
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let cover_id = Uuid::new_v4();
//! let cover_res = client
//!     .cover()
//!     .get()
//!     .cover_id(&cover_id)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("cover: {:?}", cover_res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::types::RelationshipType;
use crate::v5::schema::CoverResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned")]
pub struct GetCover<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[serde(skip)]
    pub cover_id: &'a Uuid,

    #[builder(setter(each = "add_relationship"), default)]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET ("/cover/{}", cover_id),
    #[query] GetCover<'_>,
    #[flatten_result] CoverResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::faker::lorem::en::Sentence;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path_regex};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::{Error, HttpClient, MangaDexClient};

    #[tokio::test]
    async fn get_cover_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let cover_id = Uuid::new_v4();
        let description: String = Sentence(1..3).fake();
        let created_at: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "result": "ok",
            "response": "entity",
            "data": {
                "id": cover_id,
                "type": "cover_art",
                "attributes": {
                    "volume": "1",
                    "fileName": "1.jpg",
                    "description": description,
                    "version": 1,
                    "createdAt": created_at,
                    "updatedAt": created_at,
                },
                "relationships": []
            }
        });

        Mock::given(method("GET"))
            .and(path_regex(r"/cover/[0-9a-fA-F-]+"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .cover()
            .get()
            .cover_id(&cover_id)
            .build()?
            .send()
            .await?;

        assert_eq!(res.data.id, cover_id);
        assert_eq!(res.data.attributes.volume, Some("1".to_string()));
        assert_eq!(res.data.attributes.file_name, "1.jpg".to_string());
        assert_eq!(res.data.attributes.description, description);
        assert_eq!(res.data.attributes.version, 1);
        assert_eq!(res.data.attributes.created_at, created_at);
        assert_eq!(res.data.attributes.updated_at, Some(created_at));

        Ok(())
    }

    #[tokio::test]
    async fn get_chapter_handles_404() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let cover_id = Uuid::new_v4();
        let error_id = Uuid::new_v4();

        let response_body = json!({
            "result": "error",
            "errors": [{
                "id": error_id.to_string(),
                "status": 404,
                "title": "Not found",
                "detail": "Cover could not be found"
            }]
        });

        Mock::given(method("GET"))
            .and(path_regex(r"/cover/[0-9a-fA-F-]+"))
            .respond_with(ResponseTemplate::new(404).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .cover()
            .get()
            .cover_id(&cover_id)
            .build()?
            .send()
            .await
            .expect_err("expected error");

        if let Error::Api(errors) = res {
            assert_eq!(errors.errors.len(), 1);

            assert_eq!(errors.errors[0].id, error_id);
            assert_eq!(errors.errors[0].status, 404);
            assert_eq!(errors.errors[0].title, Some("Not found".to_string()));
            assert_eq!(
                errors.errors[0].detail,
                Some("Cover could not be found".to_string())
            );
        }

        Ok(())
    }
}
