//! Builder for the author list endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-author>
//!
//! # Examples
//!
//! ```rust
//! use mangadex_api::v5::MangaDexClient;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let author_res = client
//!     .author()
//!     .list()
//!     .name("carlo zen")
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("authors: {:?}", author_res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::types::{AuthorSortOrder, RelationshipType};
use crate::v5::schema::AuthorListResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder, Default)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), default, pattern = "owned")]
pub struct ListAuthor<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    pub limit: Option<u32>,
    pub offset: Option<u32>,
    #[serde(rename = "ids")]
    #[builder(setter(each = "add_author"))]
    pub author_ids: Vec<&'a Uuid>,
    pub name: Option<String>,
    pub order: Option<AuthorSortOrder>,
    #[builder(setter(each = "add_relationship"))]
    pub includes: Vec<RelationshipType>,
}

endpoint! {
    GET "/author",
    #[query] ListAuthor<'_>,
    #[flatten_result] AuthorListResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::faker::lorem::en::Sentence;
    use fake::faker::name::en::Name;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::types::{Language, ResponseType};
    use crate::{Error, HttpClient, MangaDexClient};

    #[tokio::test]
    async fn list_author_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let author_id = Uuid::new_v4();
        let author_name: String = Name().fake();
        let author_biography: String = Sentence(1..2).fake();
        let author_created_at: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "result": "ok",
            "response": "collection",
            "data": [
                {
                    "id": author_id,
                    "type": "author",
                    "attributes": {
                        "name": author_name,
                        "imageUrl": "",
                        "biography": {
                            "en": author_biography,
                        },
                        "version": 1,
                        "createdAt": author_created_at,
                        "updatedAt": author_created_at
                    },
                    "relationships": []
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        });

        Mock::given(method("GET"))
            .and(path("/author"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .author()
            .search()
            .limit(1u32)
            .build()?
            .send()
            .await?;

        assert_eq!(res.response, ResponseType::Collection);
        let author = &res.data[0];
        assert_eq!(author.id, author_id);
        assert_eq!(author.attributes.name, author_name);
        assert_eq!(author.attributes.image_url, Some("".to_string()));
        // TODO: Known issue: Empty biography returns [] instead of {}.
        assert_eq!(author.attributes.biography.get(&Language::English), None);
        assert_eq!(author.attributes.version, 1);
        assert_eq!(author.attributes.created_at, author_created_at);
        assert_eq!(author.attributes.updated_at, Some(author_created_at));

        Ok(())
    }

    #[tokio::test]
    #[ignore = "Handling list endpoint errors needs work/refactoring"]
    async fn list_author_handles_400() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let error_id = Uuid::new_v4();

        let response_body = json!({
            "result": "error",
            "errors": [{
                "id": error_id.to_string(),
                "status": 400,
                "title": "Invalid limit",
                "detail": "Limit must be between 1 and 100"
            }]
        });

        Mock::given(method("GET"))
            .and(path(r"/author"))
            .respond_with(ResponseTemplate::new(400).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .author()
            .search()
            .limit(0u32)
            .build()?
            .send()
            .await
            .expect_err("expected error");

        if let Error::Api(errors) = res {
            assert_eq!(errors.errors.len(), 1);

            assert_eq!(errors.errors[0].id, error_id);
            assert_eq!(errors.errors[0].status, 400);
            assert_eq!(errors.errors[0].title, Some("Invalid limit".to_string()));
            assert_eq!(
                errors.errors[0].detail,
                Some("Limit must be between 1 and 100".to_string())
            );
        }

        Ok(())
    }
}
