//! Builder for the followed manga endpoint.
//!
//! <https://api.mangadex.org/docs.html#operation/get-user-follows-manga>
//!
//! # Examples
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::{MangaDexClient, Password, Username};
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let _login_res = client
//!     .auth()
//!     .login()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter23")?)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let res = client
//!     .user()
//!     .followed_manga()
//!     .limit(1_u32)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("followed manga: {:?}", res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;

use crate::v5::schema::MangaListResponse;
use crate::HttpClientRef;

#[derive(Debug, Serialize, Clone, Builder, Default)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option), pattern = "owned", default)]
pub struct FollowedManga {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[serde(skip_serializing_if = "Option::is_none")]
    limit: Option<u32>,
    #[serde(skip_serializing_if = "Option::is_none")]
    offset: Option<u32>,
}

endpoint! {
    GET "/user/follows/manga",
    #[query auth] FollowedManga,
    MangaListResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{header, method, path_regex};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::types::{ContentRating, Demographic, Language, MangaStatus};
    use crate::v5::AuthTokens;
    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn get_followed_manga_fires_a_request_to_base_url_ungrouped() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let manga_id = Uuid::new_v4();
        let manga_title = "Test Manga".to_string();
        let created_at: DateTime<Utc> = DateTime().fake();
        let response_body = json!({
            "results": [
                {
                    "result": "ok",
                    "data": {
                        "id": manga_id,
                        "type": "manga",
                        "attributes": {
                            "title": {
                                "en": manga_title
                            },
                            "altTitles": [],
                            "description": {},
                            "isLocked": false,
                            "links": null,
                            "originalLanguage": "ja",
                            "lastVolume": null,
                            "lastChapter": null,
                            "publicationDemographic": "shoujo",
                            "status": "ongoing",
                            "year": null,
                            "contentRating": "safe",
                            "tags": [],
                            "createdAt": created_at,
                            "updatedAt": created_at,
                            "version": 1
                        },
                        "relationships": []
                    }
                }
            ],
            "limit": 1,
            "offset": 0,
            "total": 1
        });

        Mock::given(method("GET"))
            .and(path_regex(r"/user/follows/manga"))
            .and(header("Authorization", "Bearer sessiontoken"))
            .respond_with(ResponseTemplate::new(200).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .user()
            .followed_manga()
            .limit(1_u32)
            .build()?
            .send()
            .await?;

        let manga = res.results[0].as_ref().unwrap();
        assert_eq!(manga.data.id, manga_id);
        assert_eq!(
            manga.data.attributes.title.get(&Language::English).unwrap(),
            &manga_title
        );
        assert!(manga.data.attributes.alt_titles.is_empty());
        assert!(manga.data.attributes.description.is_empty());
        assert!(!manga.data.attributes.is_locked);
        assert_eq!(manga.data.attributes.links, None);
        assert_eq!(manga.data.attributes.original_language, Language::Japanese);
        assert_eq!(manga.data.attributes.last_volume, None);
        assert_eq!(manga.data.attributes.last_chapter, None);
        assert_eq!(
            manga.data.attributes.publication_demographic.unwrap(),
            Demographic::Shoujo
        );
        assert_eq!(manga.data.attributes.status, MangaStatus::Ongoing);
        assert_eq!(manga.data.attributes.year, None);
        assert_eq!(
            manga.data.attributes.content_rating.unwrap(),
            ContentRating::Safe
        );
        assert!(manga.data.attributes.tags.is_empty());
        assert_eq!(manga.data.attributes.created_at, created_at);
        assert_eq!(manga.data.attributes.updated_at.unwrap(), created_at);
        assert_eq!(manga.data.attributes.version, 1);

        Ok(())
    }
}
