//! Builder for starting an upload session.
//!
//! <https://api.mangadex.org/docs.html#operation/begin-upload-session>
//!
//! ```rust
//! use uuid::Uuid;
//!
//! use mangadex_api::v5::{MangaDexClient, Username, Password};
//! use mangadex_api::types::Language;
//!
//! # async fn run() -> anyhow::Result<()> {
//! let client = MangaDexClient::default();
//!
//! let _login_res = client
//!     .auth()
//!     .login()
//!     .username(Username::parse("myusername")?)
//!     .password(Password::parse("hunter23")?)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! let group_id = Uuid::new_v4();
//! let manga_id = Uuid::new_v4();
//! let res = client
//!     .upload()
//!     .start_session()
//!     .add_group_id(&group_id)
//!     .manga_id(&manga_id)
//!     .build()?
//!     .send()
//!     .await?;
//!
//! println!("session start: {:?}", res);
//! # Ok(())
//! # }
//! ```

use derive_builder::Builder;
use serde::Serialize;
use uuid::Uuid;

use crate::v5::schema::UploadSessionResponse;
use crate::HttpClientRef;

/// Start an upload session.
///
/// This requires authentication.
///
/// Makes a request to `POST /upload/begin`.
#[derive(Debug, Builder, Serialize, Clone)]
#[serde(rename_all = "camelCase")]
#[builder(setter(into, strip_option))]
pub struct StartUploadSession<'a> {
    /// This should never be set manually as this is only for internal use.
    #[serde(skip)]
    #[builder(pattern = "immutable")]
    pub http_client: HttpClientRef,

    #[builder(setter(each = "add_group_id"))]
    pub groups: Vec<&'a Uuid>,
    #[serde(rename = "manga")]
    pub manga_id: &'a Uuid,
}

endpoint! {
    POST "/upload/begin",
    #[body auth] StartUploadSession<'_>,
    UploadSessionResponse
}

#[cfg(test)]
mod tests {
    use chrono::{DateTime, Utc};
    use fake::faker::chrono::en::DateTime;
    use fake::Fake;
    use serde_json::json;
    use url::Url;
    use uuid::Uuid;
    use wiremock::matchers::{header, method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    use crate::types::RelationshipType;
    use crate::v5::AuthTokens;
    use crate::{HttpClient, MangaDexClient};

    #[tokio::test]
    async fn start_upload_session_fires_a_request_to_base_url() -> anyhow::Result<()> {
        let mock_server = MockServer::start().await;
        let http_client: HttpClient = HttpClient::builder()
            .base_url(Url::parse(&mock_server.uri())?)
            .auth_tokens(AuthTokens {
                session: "sessiontoken".to_string(),
                refresh: "refreshtoken".to_string(),
            })
            .build()?;
        let mangadex_client = MangaDexClient::new_with_http_client(http_client);

        let group_id = Uuid::new_v4();
        let manga_id = Uuid::new_v4();
        let session_id = Uuid::new_v4();
        let date_time: DateTime<Utc> = DateTime().fake();
        let _expected_body = json!({
            "groups": [
                group_id
            ],
            "manga": manga_id
        });
        let response_body = json!({
            "id": session_id,
            "type": "upload_session",
            "attributes": {
                "isCommitted": false,
                "isProcessed": false,
                "isDeleted": false,
                "version": 1,
                "createdAt": date_time,
                "updatedAt": date_time
            }
        });

        Mock::given(method("POST"))
            .and(path("/upload/begin"))
            .and(header("Authorization", "Bearer sessiontoken"))
            .and(header("Content-Type", "application/json"))
            // TODO: Make the request body check work.
            // .and(body_json(expected_body))
            .respond_with(ResponseTemplate::new(201).set_body_json(response_body))
            .expect(1)
            .mount(&mock_server)
            .await;

        let res = mangadex_client
            .upload()
            .start_session()
            .add_group_id(&group_id)
            .manga_id(&manga_id)
            .build()?
            .send()
            .await?;

        assert_eq!(res.id, session_id);
        assert_eq!(res.type_, RelationshipType::UploadSession);
        assert_eq!(res.attributes.is_committed, false);
        assert_eq!(res.attributes.is_processed, false);
        assert_eq!(res.attributes.is_deleted, false);
        assert_eq!(res.attributes.version, 1);
        assert_eq!(res.attributes.created_at, date_time);
        assert_eq!(res.attributes.updated_at, date_time);

        Ok(())
    }
}
